"""Task implementation."""
import json
import requests
import time
import threading
import _thread
import typing as tp
from contextlib import contextmanager
from .config import Config

from tqdm import tqdm


@contextmanager
def _timeout_manager(time, msg=''):
    timer = threading.Timer(time, lambda: _thread.interrupt_main())
    if time > 0:
        timer.start()
    try:
        yield
    except KeyboardInterrupt:
        raise TimeoutError(f"Operation {msg} timeout.")
    finally:
        timer.cancel()


class Task(object):
    """Implements Task class."""

    def __init__(
        self,
        lpmn: str,
    ) -> None:
        """Constructor.

        Args:
            lpmn (str): request in lpmn notation

        """
        self.lpmn = lpmn
        self.email = Config.get("email")
        self.base_url = Config.get("base_url")
        self.output_dir = Config.get("output_path")
        self.current_task = None

    def get_progress(self) -> float:
        """Returns progress of the running task.

        Returns:
            float. A progress (value between 0 and 1).

        """
        if not self.task_id:
            return 0

        response = requests.get(
            url=f"{self.base_url}/getStatus/{self.task_id}"
        ).json()

        status = response["status"]
        if status == "ERROR":
            raise Exception(f"ERROR ({response})")
        elif status == "QUEUE":
            progress = 0
        elif status == "PROCESSING":
            progress = response["value"]
        elif status == "CANCEL":
            raise Exception("Task has been canceled")
        else:
            progress = 1

        return progress

    def get_output_id(self) -> str:
        """Returns FileID (result of completed task).

        Returns:
            FileID. FileID of finished task.

        """
        if self.get_progress() < 1:
            raise Exception("Task is not completed yet")

        response = requests.get(
            url=f"{self.base_url}/getStatus/{self.task_id}"
        ).json()

        return response["value"][0]["fileID"]

    def _wait(self, timeout=0, verbose: bool = False) -> None:
        """Wait for task to be finished.

        Args:
            timeout (float): Maximum timeout.
                Pass "0" to wait forever.
            verbose (bool): Print progress or not.

        """
        progress = self.get_progress()
        pbar = tqdm(total=100) if verbose else None

        while progress < 1:
            time.sleep(0.5)
            progress = self.get_progress()
            if pbar:
                pbar.update(progress * 100 - pbar.n)

        if pbar:
            pbar.update(100 - pbar.n)
        self.status = "done"

    def run(
        self,
        file_id: str,
        blocking: bool = True,
        timeout: float = 0,
        verbose: bool = False
    ) -> tp.Optional[str]:
        """Run task.

        Args:
            file_id (FileID): FileID of a data task should be run on.
            blocking (bool): If the call of this funciton should be blocking.
            timeout (float): Timeout used if blocking=True.
                Pass '0' if the function should wait until the task is
                finished.
            verbose (bool): Print progress or not.

        """
        prefix = f"filezip({file_id})|"
        suffix = "|dir|makezip"
        if "|dir|" in self.lpmn or self.lpmn.endswith("|dir"):
            suffix = "|makezip"

        data = json.dumps({
            "lpmn": prefix + self.lpmn + suffix,
            "Content-Type": "application/json",
            "user": self.email,
            "application": "lpmn_client",
        })

        response = requests.post(
            url=f"{self.base_url}/startTask/",
            data=data,
        )
        self.task_id = response.text

        if blocking:
            with _timeout_manager(timeout, self.lpmn):
                self._wait(timeout=timeout, verbose=verbose)
            return self.get_output_id()
