"""Basic file operations implementation."""
import io
import os
import requests
import typing as tp
import zipfile

from .config import Config


FileID = str


def file_generator(filename: str) -> str:
    """Yields all file paths from directory.

    If filename is a file (not a directory) yields filename.

    """
    if os.path.isdir(filename):
        for root, dirs, files in os.walk(filename):
            for name in files:
                yield os.path.join(root, name)
    else:
        yield filename


def upload_file(file_path: str) -> FileID:
    """Uploads file to the server.

    Args:
        file_path (str): Path of a file/directory to upload
    Returns:
        FileID. The ID of a file on a remote server.
            It can be used later for running task or downloading the file.

    """
    if not zipfile.is_zipfile(file_path):
        zip_buffer = io.BytesIO()
        with zipfile.ZipFile(
            zip_buffer,
            mode="a",
            compression=zipfile.ZIP_DEFLATED,
        ) as zip_file:
            for filename in file_generator(file_path):
                zip_file.write(
                    filename
                )
        zip_buffer.seek(0)
        files = {'file': zip_buffer}
    else:
        files = {'file': open(file_path, 'rb')}

    req = requests.post(
        url=Config.get("base_url") + "/upload/",
        files=files,
    )
    return req.text


def download_file(
    file_id: FileID,
    output_path: tp.Optional[str] = None,
    filename: tp.Optional[str] = None
) -> str:
    """Downloads file by its FileID.

    Args:
        file_id (FileID): File id of a file to download
        output_path (Optional[str]): Directory to save downloaded file in.
            If not given the file will be saved in default location
            (config.ini).
        filename (Optional[str]): Output filename without .zip format.
            If not specified, filename is random.
    Returns:
        str. A path to downloaded file.

    """
    response = requests.get(url=f"{Config.get('base_url')}/download{file_id}")
    if response.status_code == 404 or response.status_code == 500:
        raise FileNotFoundError("File not found")
    if not filename:
        filename = f"{file_id.split('/')[-1]}.zip"
    if not output_path:
        output_path = Config.get("output_path")
    os.makedirs(output_path, exist_ok=True)

    output_path = f"{output_path}/{filename}"
    with open(output_path, "wb") as f:
        f.write(response.content)
    return output_path


def download_file_as_dict(
    file_id: FileID,
) -> tp.Dict[str, str]:
    """Downloads file by its FileID.

    Args:
        file_id (FileID): File id of a file to download

    Returns:
        dict. Dictionary mapping filenames in output zipfile to file contents.

    """
    def try_decode(bytes_data):
        try:
            bytes_data.decode("utf-8")
            return True
        except UnicodeDecodeError:
            return False

    response = requests.get(url=f"{Config.get('base_url')}/download{file_id}")
    if response.status_code == 404 or response.status_code == 500:
        raise FileNotFoundError("File not found")

    zip_buffer = io.BytesIO(initial_bytes=response.content)
    with zipfile.ZipFile(
        zip_buffer,
        mode="r",
        compression=zipfile.ZIP_DEFLATED,
    ) as zip_file:
        return {
            i.filename: zip_file.open(i).read().decode("utf-8")
            for i in filter(
                lambda x:
                    try_decode(zip_file.read(x)) and not x.is_dir(),
                zip_file.infolist()
            )
        }
