"""Config implementation."""
import configparser
import logging
import os

BASE_DIR = (
    os.getenv('APPDATA') if os.name == "nt" else os.path.expanduser('~/.config')
)
USER_CONFIG = os.path.join(BASE_DIR, "lpmn_client/config.ini")


def replace_with_envs(config):
    """Replaces config values with enviroment variables."""
    possible_values = {
        "LPMN_EMAIL": "email",
        "LPMN_OUT": "output_path"
    }
    env_values = {
        possible_values[k]: v for k, v in os.environ.items()
        if k in possible_values.keys()
    }
    for k, v in env_values.items():
        config["APP"][k] = v

    return config


def parse_config():
    """Parse config."""
    defaults = {
        'APP': {
            'base_url': 'http://ws.clarin-pl.eu/nlprest2/base',
            'output_path': './out',
            'email': 'lpmn_client@clarin-eu.pl'
        }
    }
    config = configparser.ConfigParser()
    config.read_dict(defaults)

    if os.path.isfile("config.ini"):
        config.read("config.ini")
    elif os.path.isfile(USER_CONFIG):
        config.read(USER_CONFIG)
    config = replace_with_envs(config)

    if config["APP"]["email"] == defaults["APP"]["email"]:
        logging.warning(
            "Email not provided - using default value. "
            "Please, visit https://gitlab.clarin-pl.eu/nlpworkers/lpmn_client "
            "for more information."
        )

    return config


class Config:
    """Config class."""

    __attrs = parse_config()["APP"]

    @staticmethod
    def get(key: str):
        """Returns value by its key form config.

        Args:
            key: any value

        Returns:
            value under key

        """
        return Config.__attrs[key]

    @staticmethod
    def set(key, value):
        """Sets value using key."""
        Config.__attrs[key] = value
