"""Replacer for user mentions.""" ""

from typing import List, Tuple

from random_username.generate import generate_username

from src.detections import Detection, UserDetection
from src.replacers.interface import ReplacerInterface
from src.string_replacements import replace_and_update


class UserReplacer(ReplacerInterface):
    """Replaces any detected user mentions with random username."""

    def replace(
        self, text: str, detections: List[Tuple[int, int, Detection]]
    ) -> Tuple[str, List[Tuple[int, int, Detection]]]:
        """Replace detected user mentions in text with anonimized version.

        Eg.: "@bob213" -> "@beautifullion56"

        Args:
            text (str): Text to be processed.
            detections (List[Tuple[int, int, str]]): List of detections.

        Returns:
            Tuple[str, List[Tuple[int, int, str]]]: Text with supported entities
            replaced with anonimized version and list of detections that were
            not processed by this replacer.

        """
        replacements = []
        not_processed = []

        already_replaced = dict()

        for item in detections:
            start, end, detection = item

            if isinstance(detection, UserDetection):
                if text[start:end] not in already_replaced:
                    username = "@" + generate_username(1)[0]
                    already_replaced[text[start:end]] = username

                replacements.append((start, end, already_replaced[text[start:end]]))
            else:
                not_processed.append(item)

        return replace_and_update(text, replacements, not_processed)
