"""Tag Replacer module."""

from typing import List, Tuple

from src.detections import (
    CityDetection,
    CountryDetection,
    DateDetection,
    Detection,
    EmailDetection,
    HydronymDetection,
    KRSDetection,
    LocationDetection,
    NameDetection,
    NumberDetection,
    OrganizationNameDetection,
    PhoneNumberDetection,
    ProperNameDetection,
    SerialNumberDetection,
    StreetNameDetection,
    SurnameDetection,
    TINDetection,
    TitleDetection,
    UrlDetection,
    UserDetection,
)
from src.replacers.interface import ReplacerInterface
from src.string_replacements import replace


class TagReplacer(ReplacerInterface):
    """Replaces any detected entities with tags.

    Example: Anna has a cat -> [PERSON] has a cat.

    """

    def __init__(self):
        """Inits tag replacer."""
        # TODO: Move this to a config file.
        self.tags_map = {
            NameDetection: "[OSOBA]",
            SurnameDetection: "[OSOBA]",
            StreetNameDetection: "[MIEJSCE]",
            LocationDetection: "[MIEJSCE]",
            CityDetection: "[MIEJSCE]",
            CountryDetection: "[MIEJSCE]",
            OrganizationNameDetection: "[ORGANIZACJA]",
            ProperNameDetection: "[NAZWA WŁASNA]",
            TitleDetection: "[TYTUŁ]",
            PhoneNumberDetection: "[CYFRY]",
            HydronymDetection: "[NAZWA WODNA]",
            UrlDetection: "[WWW]",
            UserDetection: "@[USER]",
            EmailDetection: "[MAIL]",
            DateDetection: "[DATA]",
            TINDetection: "[CYFRY]",
            KRSDetection: "[CYFRY]",
            SerialNumberDetection: "[NUMER IDENTYFIKACYJNY]",
            NumberDetection: "[CYFRY]",
        }

    def replace(
        self, text: str, detections: List[Tuple[int, int, Detection]]
    ) -> Tuple[str, List[Tuple[int, int, Detection]]]:
        """Replace detected entities in text with a tag like [CITY].

        Args:
            text (str): Text to be processed.
            detections (List[Tuple[int, int, str]]): List of detections.

        Returns:
            Tuple[str, List[Tuple[int, int, str]]]: Text with supported entities
            replaced with tags and list of detections that were
            not processed by this replacer.

        """
        result = [
            (start, end, self.tags_map.get(type(entity_type), "[OTHER]"))
            for start, end, entity_type in detections
        ]

        return replace(text, result), []
