"""Module responsible for replacing numbers with random digits."""

import random
import string
from typing import List, Tuple

from src.detections import Detection, NumberDetection
from src.replacers.interface import ReplacerInterface
from src.string_replacements import replace_and_update


def _randomize_digits_in_text(text: str) -> str:
    result = ""

    for c in text:
        if c.isdigit():
            result += random.choice(string.digits)
        else:
            result += c

    return result


class NumberReplacer(ReplacerInterface):
    """Replaces any detected numbers with random digits."""

    def replace(
        self, text: str, detections: List[Tuple[int, int, Detection]]
    ) -> Tuple[str, List[Tuple[int, int, Detection]]]:
        """Replace detected numbers in text with anonimized version.

        Eg.: "123456789" -> "692154236"

        Args:
            text (str): Text to be processed.
            detections (List[Tuple[int, int, str]]): List of detections.

        Returns:
            Tuple[str, List[Tuple[int, int, str]]]: Text with supported entities
            replaced with anonimized version and list of detections that were
            not processed by this replacer.

        """
        replacements = []
        not_processed = []

        already_replaced = dict()

        for item in detections:
            start, end, detection = item

            if isinstance(detection, NumberDetection):
                if text[start:end] not in already_replaced:
                    already_replaced[text[start:end]] = _randomize_digits_in_text(
                        text[start:end]
                    )

                replacements.append((start, end, already_replaced[text[start:end]]))
            else:
                not_processed.append(item)

        return replace_and_update(text, replacements, not_processed)
