"""Module for replacing NER entities with anonimized version."""

from typing import List, Tuple
from collections import defaultdict
from src.detections import Detection, MorphosyntacticInfoMixin
from src.dictionaries.morphosyntactic import MorphosyntacticDictionary
from src.replacers.interface import ReplacerInterface
from src.string_replacements import replace_and_update


class NERReplacer(ReplacerInterface):
    """Replaces any detected NER entities with their morphosyntactic equivalents."""

    def __init__(self, dictionary: MorphosyntacticDictionary):
        """Initializes NERReplacer.

        Args:
            dictionary (MorphosyntacticDictionary): Dictionary used for
                replacing words with their morphosyntactic equivalents.

        """
        self._dictionary = dictionary

    def replace(
        self, text: str, detections: List[Tuple[int, int, Detection]]
    ) -> Tuple[str, List[Tuple[int, int, Detection]]]:
        """Replace detected NER entities in text with anonimized version.

        Eg.: "John Smith is a nice guy" -> "Mark Johnson is a nice guy"

        Args:
            text (str): Text to be processed.
            detections (List[Tuple[int, int, str]]): List of detections.

        Returns:
            Tuple[str, List[Tuple[int, int, str]]]: Text with supported entities
            replaced with anonimized version and list of detections that were
            not processed by this replacer.

        """
        replacements = []
        not_processed = []

        morpho_detections = defaultdict(list)
        non_morpho_detections = defaultdict(list)

        for detection in detections:
            start, end, detection_entry = detection

            if (
                type(detection_entry)
                not in self._dictionary.get_supported_detection_classes()
            ):
                not_processed.append(detection)
                continue

            if isinstance(detection_entry, MorphosyntacticInfoMixin):
                key = (
                    detection_entry.lemma or text[start:end],
                    detection_entry.TYPE_NAME,
                )
                morpho_detections[key].append(detection)
            else:
                key = (text[start:end], detection.TYPE_NAME)
                non_morpho_detections[key].append(detection)

        # Replace morphosyntactic detections
        for key, detections in morpho_detections.items():
            detection_entries = [detection[2] for detection in detections]
            detection_replacements = self._dictionary.get_random_replacements(
                detection_entries
            )

            for detection, replacement in zip(detections, detection_replacements):
                start, end, detection = detection
                replacement_word = replacement["word"]

                if replacement_word is None:
                    not_processed.append((start, end, detection))
                else:
                    replacements.append((start, end, replacement_word))

        # Replace non-morphosyntactic detections
        for key, detections in non_morpho_detections.items():
            replacement = self._dictionary.get_random_replacement(detections[0][2])

            for detection in detections:
                start, end, detection = detection
                replacement_word = replacement["word"]

                if replacement_word is None:
                    not_processed.append((start, end, detection))
                else:
                    replacements.append((start, end, replacement_word))

        return replace_and_update(text, replacements, not_processed)
