"""Email replacer module."""

import random
import string
from typing import List, Tuple

from src.detections import Detection, EmailDetection
from src.replacers.interface import ReplacerInterface
from src.string_replacements import replace_and_update


def _random_char(char_num):
    return "".join(random.choice(string.ascii_letters) for _ in range(char_num))


def _random_email():
    return _random_char(7) + "@email.com"


class EmailReplacer(ReplacerInterface):
    """Replaces any detected emails with random email."""

    def replace(
        self, text: str, detections: List[Tuple[int, int, Detection]]
    ) -> Tuple[str, List[Tuple[int, int, Detection]]]:
        """Replace detected emails in text with anonimized version.

        eg.: "My email is bob@shop.com -> My email is 532fasfs@email.com"

        Args:
            text (str): Text to be processed.
            detections (List[Tuple[int, int, str]]): List of detections.

        Returns:
            Tuple[str, List[Tuple[int, int, str]]]: Text with supported entities
            replaced with anonimized version and list of detections that were
            not processed by this replacer.

        """
        replacements = []
        not_processed = []

        already_replaced = dict()

        for item in detections:
            start, end, detection = item

            if isinstance(detection, EmailDetection):
                if text[start:end] not in already_replaced:
                    already_replaced[text[start:end]] = _random_email()

                replacements.append((start, end, already_replaced[text[start:end]]))
            else:
                not_processed.append(item)

        return replace_and_update(text, replacements, not_processed)
