import json
import uuid
from pathlib import Path
from typing import Any
from google.protobuf.json_format import MessageToJson

from google.cloud.storage import Blob
import os
from google.cloud import speech
from google.cloud import storage


class GoogleCloudAsrProcessor:
    bucket_name: str
    storage_client: storage.Client
    speech_client: speech.SpeechClient
    language_code: str

    def __init__(
        self,
        bucket_name: str,
        storage_client: storage.Client,
        speech_client: speech.SpeechClient,
        language_code: str
    ):
        self.speech_client = speech_client
        self.storage_client = storage_client
        self.bucket_name = bucket_name
        self.language_code = language_code

    def save_file_in_gcs(self, file_path: str) -> Blob:
        bucket = self.storage_client.get_bucket(self.bucket_name)
        blob = bucket.blob(f'{uuid.uuid4()}__{Path(file_path).name}')
        blob.upload_from_filename(file_path)
        return blob

    def process_file(self, file_path: str) -> Any:
        blob = self.save_file_in_gcs(file_path)
        uri = f'gs://{self.bucket_name}/{blob.name}'
        long_audi_wav = speech.RecognitionAudio(uri=uri)
        config_wav_enhanced = speech.RecognitionConfig(
            enable_automatic_punctuation=True,
            language_code=self.language_code,
            use_enhanced=True,
            enable_word_time_offsets=True
        )

        operation = self.speech_client.long_running_recognize(
            config=config_wav_enhanced,
            audio=long_audi_wav
        )
        response = operation.result(timeout=300)
        blob.delete()
        return MessageToJson(response._pb)


if __name__ == '__main__':
    os.environ['GOOGLE_APPLICATION_CREDENTIALS'] = '/home/marcinwatroba/PWR_ASR/asr-benchmarks/secret_key.json'
    media_file_name_wav = 'experiment_data/dataset/voicelab_cbiz_testset_20220322/bankowe/cbiz_tc_2.agnt.wav'
    processor = GoogleCloudAsrProcessor('asr-benchmarks-data', storage.Client(), speech.SpeechClient(), 'pl-PL')
    processor.process_file(media_file_name_wav)

