# coding: utf8

# Copyright (C) 2017 Michał Kaliński
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Those tuples are returned by readers and absorbed by storages."""

from collections import namedtuple


__all__ = (
    'SynsetNode',
    'LexicalUnitNode',
    'RelationTypeNode',
    'make_synset_node',
    'make_lexical_unit_node',
    'make_relation_type_node',
)


SynsetNode = namedtuple(
    "SynsetNode",
    ["id", "legacy_id", "definition", "related", "is_artificial"],
)
LexicalUnitNode = namedtuple(
    "LexicalUnitNode",
    ["id", "legacy_id", "lemma", "pos", "variant", "synset", "unit_index",
     "definition", "usage_notes", "external_links", "examples",
     "examples_sources", "domain", "related", "verb_aspect", "is_emotional",
     "emotion_markedness", "emotion_names", "emotion_valuations",
     "emotion_example_1", "emotion_example_2"]
)
RelationTypeNode = namedtuple(
    "RelationTypeNode",
    ["kind", "legacy_id", "name", "parent", "aliases"],
)


def make_synset_node(**props):
    """Create a :class:`SynsetNode` instance.

    Inserting appropriate "empty" values into optional properties
    where a value was not passed.

    Takes only keyword arguments, and passes them to :class:`SynsetNode`
    constructor.
    """
    syn = SynsetNode(
        id=props.pop('id'),
        legacy_id=props.pop('legacy_id', None),
        definition=props.pop('definition', None),
        related=props.pop('related', ()),
        is_artificial=props.pop('is_artificial', False),
    )

    if props:
        raise KeyError('Not known synset properties: ' + repr(tuple(props)))

    return syn


def make_lexical_unit_node(**props):
    """Create a :class:`LexicalUnitNode` instance.

    Inserting appropriate "empty" values into optional properties
    where a value was not passed.

    Takes only keyword arguments, and passes them to :class:`LexicalUnitNode`
    constructor.
    """
    lex = LexicalUnitNode(
        id=props.pop('id'),
        legacy_id=props.pop('legacy_id', None),
        lemma=props.pop('lemma'),
        pos=props.pop('pos'),
        variant=props.pop('variant'),
        synset=props.pop('synset'),
        unit_index=props.pop('unit_index'),
        definition=props.pop('definition', None),
        usage_notes=props.pop('usage_notes', ()),
        external_links=props.pop('external_links', ()),
        examples=props.pop('examples', ()),
        examples_sources=props.pop('examples_sources', ()),
        domain=props.pop('domain'),
        related=props.pop('related', ()),
        verb_aspect=props.pop('verb_aspect', None),
        is_emotional=props.pop('is_emotional', None),
        emotion_markedness=props.pop('emotion_markedness', None),
        emotion_names=props.pop('emotion_names', ()),
        emotion_valuations=props.pop('emotion_valuations', ()),
        emotion_example_1=props.pop('emotion_example_1', None),
        emotion_example_2=props.pop('emotion_example_2', None),
    )

    if props:
        raise KeyError(
            'Not known lexical unit properties: ' + repr(tuple(props)),
        )

    return lex


def make_relation_type_node(**props):
    """Create a :class:`RelationNode` instance.

    Inserting appropriate "empty" values into optional properties
    where a value was not passed.

    Takes only keyword arguments, and passes them to :class:`RelationNode`
    constructor.
    """
    rel = RelationTypeNode(
        name=props.pop('name'),
        legacy_id=props.pop('legacy_id', None),
        kind=props.pop('kind'),
        parent=props.pop('parent', None),
        aliases=props.pop('aliases', frozenset()),
    )

    if props:
        raise KeyError(
            'Not known relation properties: ' + repr(tuple(props)),
        )

    return rel
