# coding: utf8

# Copyright (C) 2017 Michał Kaliński
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Parsing strings in wordnet comment format.

For readers that need to deal with them.

Importing this module introduces dependency on wncomments.
"""

from __future__ import absolute_import, division


from collections import namedtuple
import itertools as itt

import plwn_comments as plwnc
import plwn_comments.exceptions as plwnce
import plwn_comments.utils.usage_tags as plwncu


__all__ = (
    'WN_TAGS',
    'NON_EXAMPLE_TAG_NAMES',
    'CommentData',
    'parse_comment_string',
    'make_empty_comment_data',
)


#: :class:`plwn_comments.TagBank` structure that defines all kinds of comment
#: tags which are needed by PLWN API.
WN_TAGS = plwnc.TagBank()
# Usage notes
WN_TAGS.define(u'K')
# External links
WN_TAGS.define(u'L', u'{')
# Definition
WN_TAGS.define(u'D')

#: The distinction for these tags is useful, since all examples go to one
#: place.
NON_EXAMPLE_TAG_NAMES = frozenset((u'K', u'L', u'D'))

# And define those example tags
WN_TAGS.define_from(
    plwncu.iter_usage_tags(),
    plwncu.DEFAULT_USAGE_TAG_SURROUND,
)

#: Data tuple returned from :func:`parse_comment_string`.
CommentData = namedtuple(
    'CommentData',
    ('examples', 'examples_sources', 'definition', 'usage', 'links'),
)


def parse_comment_string(cmt_str):
    """Parse a comment string.

    Extract all data required by PLWN API packed in a named tuple.

    :param str cmt_str: String in PLWN comment format.

    :returns: Extracted and ordered items needed by PLWN API.
    :rtype: CommentData
    """
    try:
        cmt = plwnc.Comment.parse(cmt_str, WN_TAGS)
    except plwnce.PLWNCommentsException:
        # For now just make an empty comment which will make all fields unset
        cmt = plwnc.Comment(WN_TAGS)

    # Get all examples
    examples = []
    examples_src = []

    for tagname, tagcontents in cmt.items():
        if tagname not in NON_EXAMPLE_TAG_NAMES:
            examples.extend(tagcontents)
            examples_src.extend(itt.repeat(tagname, len(tagcontents)))

    return CommentData(
        tuple(examples),
        tuple(examples_src),
        cmt.get_first(u'D', None),
        tuple(cmt[u'K']),
        tuple(cmt[u'L']),
    )


def make_empty_comment_data():
    """Create an empty ``CommentData`` instance.

    For cases where there's no comment to parse but an instance
    with null-data is needed.
    """
    return CommentData(
        examples=(),
        examples_sources=(),
        definition=None,
        usage=(),
        links=(),
    )
