# coding: utf8

# Copyright (C) 2017 Michał Kaliński
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Custom exceptions raised by PLWN API."""

from __future__ import absolute_import, division


__all__ = (
    'PLWNAPIException',
    'NotFound',
    'LexicalUnitNotFound',
    'SynsetNotFound',
    'ReaderException',
    'MalformedIdentifierException',
    'LoadException',
    'DumpVersionException',
    'InvalidRelationTypeException',
    'AmbiguousRelationTypeException',
)


class PLWNAPIException(Exception):
    """Base for all exceptions in the module."""

    pass


class NotFound(PLWNAPIException):
    """Base for exceptions raised when an entity is not found."""

    pass


class LexicalUnitNotFound(NotFound):
    """Raised when a lexical unit is not found during lookup."""

    pass


class SynsetNotFound(NotFound):
    """Raised when a synset is not found during lookup."""

    pass


class ReaderException(PLWNAPIException):
    """Raised when there's an error in the format expected by a reader."""

    pass


class MalformedIdentifierException(ReaderException):
    """Raised during UBY-LMF parsing.

    When a malformed identifier is encountered.
    """

    def __init__(self, id_):
        super(MalformedIdentifierException, self).__init__(
            "Malformed identifier, expected digits at the end of the original"
            " id instead got {!r}"
            .format(id_)
        )


class LoadException(PLWNAPIException):
    """Raised when a storage can't be loaded from file."""

    pass


class DumpVersionException(LoadException):
    """Raised when a dumped storage has wrong version.

    Suggesting incompatible format.
    """

    def __init__(self, version_is, version_required):
        super(DumpVersionException, self).__init__(version_is,
                                                   version_required)
        self.version_is = version_is
        self.version_required = version_required

    def __str__(self):
        return (
            'Invalid schema version of dumped storage: {!r} (should be {!r})'
            .format(self.version_is, self.version_required)
        )


class InvalidRelationTypeException(PLWNAPIException):
    """Raised when relation identifier does not refer to any existing relation.

    Or the relation exists for the other relation kind.
    """

    pass


class AmbiguousRelationTypeException(InvalidRelationTypeException):
    """.

    Raised when a relation type identifier could refer to more than
    one relation, but only one is permitted in the context.
    """

    pass
