# coding: utf8

# Copyright (C) 2017 Michał Kaliński
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""All enumerated values used in plWordNet."""

from __future__ import absolute_import, division


import re
from enum import Enum

import six


__all__ = (
    'RelationKind',
    'PoS',
    'VerbAspect',
    'EmotionMarkedness',
    'EmotionName',
    'EmotionValuation',
    'Domain',
    'make_values_tuple',
)


def _numtrans(n2e_dict, e2n_dict, start_value=1):
    """Helper decorator for making dictionaries.

    Translating enum instances into numbers denoting them in plWN database.
    """

    def decorator(cls):
        def by_db_number(number, optional=False):
            """Return the enum value.

            Associated with ``number`` value stored in the plWordNet database.

            Raises ``KeyError`` if ``number`` is not in the range valid for
            the database field, unless ``optional`` is ``True``; then,
            ``None`` is returned instead of an enum value.
            """
            try:
                return n2e_dict[number]
            except KeyError:
                if optional:
                    return None
                raise

        def db_number(self):
            """Number associated with the enum value in plWordNet database."""
            return e2n_dict[self]

        cls.by_db_number = staticmethod(by_db_number)
        cls.db_number = property(db_number)

        for num, enval in enumerate(cls, start_value):
            n2e_dict[num] = enval
            e2n_dict[enval] = num

        return cls
    return decorator


_RK_NUM2ENUM = {}
_RK_ENUM2NUM = {}


@_numtrans(_RK_NUM2ENUM, _RK_ENUM2NUM, 0)
class RelationKind(Enum):
    """Whether a relation connects synsets or lexical units."""

    # Explicit ordering is needed only in python 2.
    if six.PY2:
        __order__ = 'lexical synset'

    lexical = u'lexical'
    synset = u'synset'


_POS_NUM2ENUM = {}
_POS_ENUM2NUM = {}


@_numtrans(_POS_NUM2ENUM, _POS_ENUM2NUM)
class PoS(Enum):
    """Defines part of speech values used in plWordNet."""

    if six.PY2:
        __order__ = 'verb noun adverb adjective ' \
            'verb_en noun_en adverb_en adjective_en'

    verb = u'czasownik'
    noun = u'rzeczownik'
    adverb = u'przysłówek'
    adjective = u'przymiotnik'

    # English (PWN) PoSes
    verb_en = u'verb_en'
    noun_en = u'noun_en'
    adverb_en = u'adverb_en'
    adjective_en = u'adjective_en'

    v = verb
    n = noun
    adv = adverb
    adj = adjective

    v_en = verb_en
    n_en = noun_en
    adv_en = adverb_en
    adj_en = adjective_en

    # Polish aliases for parity
    verb_pl = verb
    noun_pl = noun
    adverb_pl = adverb
    adjective_pl = adjective

    v_pl = verb
    n_pl = noun
    adv_pl = adverb
    adj_pl = adjective

    @property
    def short_value(self):
        """Returns shortened value.

        With "adjective" shortened to "adj" and "adverb" to "adv"
        for compatibility with other conventions.
        """
        if not hasattr(self, "_short_value"):
            value = self.value
            value = value.replace("adjective", "adj")
            value = value.replace("adverb", "adv")
            self._short_value = value
        return self._short_value

    @property
    def is_polish(self):
        return not self.is_english

    @property
    def is_english(self):
        if not hasattr(self, "_is_english"):
            self._is_english = self.value.endswith("_en")
        return self._is_english


_VA_NUM2ENUM = {}
_VA_ENUM2NUM = {}


@_numtrans(_VA_NUM2ENUM, _VA_ENUM2NUM)
class VerbAspect(Enum):
    """Defines verb aspect values used in plWordNet."""

    if six.PY2:
        __order__ = 'perfective imperfective predicative two_aspect no'

    no = u'no'
    perfective = u'perf'
    imperfective = u'imperf'
    predicative = u'pred'
    two_aspect = u'imperf.perf'

    perf = perfective
    imperf = imperfective
    pred = predicative
    two = two_aspect

    # Additionally, some Polish abbreviations
    dk = perfective
    ndk = imperfective


class EmotionMarkedness(Enum):
    """Defines markedness of emotions associated with some lexical units."""

    strong_positive = u'+ m'
    strong_negative = u'- m'
    weak_positive = u'+ s'
    weak_negative = u'- s'
    ambiguous = u'amb'

    plus_m = strong_positive
    minus_m = strong_negative
    plus_s = weak_positive
    minus_s = weak_negative
    amb = ambiguous

    @classmethod
    def normalized(cls, strvalue):
        """Return an instance of this enum.

        Corresponding to ``strvalue`` after normalizing it with regards
        to whitespace.
        """
        strvalue = strvalue.strip()

        # Try the one value value that won't require matching
        if strvalue == cls.ambiguous.value:
            return cls.ambiguous

        match = re.match(r'([+-])\s*([sm])', strvalue, re.U)

        if not match:
            # This can't be a valid string, so let the built-in exception
            # raise.
            return cls(strvalue)

        return cls(match.group(1) + u' ' + match.group(2))


class EmotionName(Enum):
    """Defines names of emotions that may be associated with lexical units."""

    joy = u'radość'
    trust = u'zaufanie'
    anticipation = u'cieszenie się na coś oczekiwanego'
    surprise = u'zaskoczenie czymś nieprzewidywanym'
    sadness = u'smutek'
    anger = u'złość'
    fear = u'strach'
    disgust = u'wstręt'

    radosc = joy
    zaufanie = trust
    cieszenie_sie_na = anticipation
    zaskoczenie = surprise
    smutek = sadness
    zlosc = anger
    strach = fear
    wstret = disgust


class EmotionValuation(Enum):
    """Defines valuations of emotions.

    That may be associated with lexical units.
    """

    usefulness = u'użyteczność'
    good = u'dobro'
    truth = u'prawda'
    knowledge = u'wiedza'
    beauty = u'piękno'
    happiness = u'szczęście'
    uselessness = u'nieużyteczność'
    harm = u'krzywda'
    ignorance = u'niewiedza'
    error = u'błąd'
    ugliness = u'brzydota'
    unhappiness = u'nieszczęście'

    uzytecznosc = usefulness
    dobro = good
    prawda = truth
    wiedza = knowledge
    piekno = beauty
    szczescie = happiness
    nieuzytecznosc = uselessness
    krzywda = harm
    niewiedza = ignorance
    blad = error
    brzydota = ugliness
    nieszczescie = unhappiness


_DOM_NUM2ENUM = {}
_DOM_ENUM2NUM = {}


@_numtrans(_DOM_NUM2ENUM, _DOM_ENUM2NUM)
class Domain(Enum):
    """Defines domains of lexical units occurring in plWordNet."""

    if six.PY2:
        __order__ = 'bhp czy wytw cech czc umy por zdarz czuj jedz grp msc ' \
            'cel rz os zj rsl pos prc il zw ksz st sbst czas zwz hig zmn ' \
            'cumy cpor wal cjedz dtk cwytw cczuj ruch pst cpos sp cst pog ' \
            'jak rel odcz grad sys adj adv cdystr caku cper cdel'

    bhp = u'najwyższe w hierarchii'
    czy = u'czynności (nazwy)'
    wytw = u'wytwory ludzkie(nazwy)'
    cech = u'cechy ludzi i zwierząt'
    czc = u'części ciała'
    umy = u'związane z myśleniem'
    por = u'związane z porozumiewaniem się'
    zdarz = u'zdarzenia'
    czuj = u'uczucia, odczucia i emocje'
    jedz = u'jedzenie'
    grp = u'grupy ludzi i rzeczy'
    msc = u'miejsca i umiejscowienie'
    cel = u'cel działania'
    rz = u'obiekty naturalne'
    os = u'ludzie'
    zj = u'zjawiska naturalne'
    rsl = u'nazwy roślin'
    pos = u'posiadanie i jego zmiana'
    prc = u'procesy naturalne'
    il = u'ilość, liczebność, jednoski miary'
    zw = u'zwierzęta'
    ksz = u'kształty'
    st = u'sytuacje statyczne (stany)'
    sbst = u'substancje'
    czas = u'czas i stosunki czasowe'
    zwz = u'związek miedzy ludźmi, rzeczami lub ideami'

    hig = u'pielęgnacja ciała'
    zmn = u'zmiana wielkości, temeraturym natężenia, itp.'
    cumy = u'czasowniki myślenia (szeroko rozumianego)'
    cpor = u'czasowniki mówienia, śpiewania itp.'
    wal = u'czasowniki rywalizacji fizycznej'
    cjedz = u'czasowniki jedzenia'
    dtk = u'czasowniki oznacz. kontakt fizyczny ' \
        u'(dotykanie, uderzenie, rycie itp.)'
    cwytw = u'czasowniki oznacz. wytwarzanie czegoś'
    cwyt = u'czasowniki oznacz. wytwarzanie czegoś'
    cczuj = u'czasowniki wyrażające uczucia'
    ruch = u'czasowniki ruchu'
    pst = u'czasowniki postrzegania (percepcji)'
    cpos = u'czasowniki posiadania i zmiany posiadania'
    sp = u'czasowniki oznacz. wydarzenie i działania społeczne i polityczne'
    cst = u'czasowniki stanowe'
    pog = u'czasowniki oznacz. zjawiska pogodowe'

    jak = u'przymiotniki jakościowe'
    rel = u'przymiotniki relacyjne (rzeczownikowe)'
    odcz = u'przymiotniki odczasownikowe'
    grad = u'przymiotniki odprzymiotnikowe (natężenie cechy)'

    sys = u'systematyka, klasyfikacja'

    adj = u'all adjective clusters'
    adv = u'all adverbs'

    mat = u'przymiotniki materiałowe'

    cdystr = u'czasownki dystrybutywne'
    caku = u'czasowniki akumulatywne'
    cper = u'czasowniki perduratywne'
    cdel = u'czasowniki delimitatywne'


def make_values_tuple(enum_seq):
    """Auxiliary function.

    That converts a sequence of enums to a tuple of enum string values.
    """
    return tuple(en.value for en in enum_seq)
