# coding: utf8

# Copyright (C) 2017 Michał Kaliński
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Base, abstract classes for plWordNet objects.

Implementing common functionality independent of structures
holding the data itself.
"""

from __future__ import absolute_import, division

from abc import ABCMeta, abstractmethod, abstractproperty
from collections import namedtuple
import locale
import operator as op

import six

from .utils import graphmlout as go
from .enums import make_values_tuple


__all__ = (
    'PLWordNetBase',
    'SynsetBase',
    'LexicalUnitBase',
    'RelationInfoBase',
    'RelationEdge',
)


#: Tuple type representing a relation instance between two synsets or lexical
#: units.
RelationEdge = namedtuple('RelationEdge', ('source', 'relation', 'target'))


@six.add_metaclass(ABCMeta)
class PLWordNetBase(object):
    """The primary object providing data from plWordNet.

    Allows retrieving synsets, lexical units, and other informative objects.
    """

    _STORAGE_NAME = '?'

    @classmethod
    def from_reader(cls, reader, dump_to=None):
        """Create new instance from a source reader.

        Optionally saving it in an internal representation format in
        another file.

        ``reader`` is any iterable that yields node instances:
        :class:`~plwn.readers.nodes.SynsetNode`,
        :class:`~plwn.readers.nodes.LexicalUnitNode` and
        :class:`~plwn.readers.nodes.RelationTypeNode`.

        ``dump_to`` is a path to a (non-existing) file where data
        form ``reader`` will be stored to be to be loaded later.
        If not passed, then the data won't be cached in any file, requiring
        to be read again using :meth:`.from_reader`.
        """
        raise NotImplementedError()

    @classmethod
    def from_dump(cls, dump):
        """Create new instance from a dump of cached internal representation.

        The dump file must have been created by :meth:`.from_reader` of the
        same :class:`PLWordNetBase` subclass and schema version.
        """
        return NotImplementedError()

    @abstractmethod
    def synsets(self, lemma=None, pos=None, variant=None):
        """Select synsets from plWordNet based on combination of criteria.

        This method works just like :meth:`.lexical_units`, but returns an
        iterable of distinct synsets that own the lexical units selected by
        the query.
        """
        pass

    @abstractmethod
    def synset(self, lemma, pos, variant):
        """Like :meth:`.synsets`.

        But either return a single synset or raise
        :exc:`~plwn.exceptions.SynsetNotFound`.

        All parameters are required, to ensure that the query could only match
        a single synset.
        """
        pass

    @abstractmethod
    def synset_by_id(self, id_):
        """Select a synset using its internal, numeric ID.

        If there is no synset with the given ID, raise
        :exc:`~plwn.exceptions.SynsetNotFound`.

        This is the fastest method to get a particular :class:`SynsetBase`
        object.
        """
        pass

    @abstractmethod
    def lexical_units(self, lemma=None, pos=None, variant=None):
        """Select lexical units from plWordNet based on combination of criteria.

        It's possible to specify the lemma, part of speech and variant of the
        units this method should yield. If a parameter value is omitted, any
        value matches. Conversely, a call of ``lexical_units()`` will return
        an iterable of all lexical units in plWordNet. If no lexical unit
        matches the query, returns an empty iterable.

        The parameter ``lemma`` is an unicode string, ``variant`` is an
        integer, and ``pos`` is an enumerated value of
        :class:`~plwn.enums.PoS`.
        """
        pass

    @abstractmethod
    def lexical_unit(self, lemma, pos, variant):
        """Like :meth:`.lexical_units`.

        But either return a single lexical unit or
        raise :exc:`~plwn.exceptions.LexicalUnitNotFound`.

        All parameters are required, to ensure that the query could only match
        a single lexical unit.
        """
        pass

    @abstractmethod
    def lexical_unit_by_id(self, id_):
        """Select a lexical unit using its internal, numeric ID.

        If there is no lexical unit with the given ID, raise
        :exc:`~plwn.exceptions.LexicalUnitNotFound`.

        This is the fastest method to get a particular :class:`LexicalUnitBase`
        object.
        """
        pass

    @abstractmethod
    def synset_relation_edges(self,
                              include=None,
                              exclude=None,
                              skip_artificial=True):
        """Get an iterable of synset relation instances from plWordNet.

        As represented by :class:`RelationEdge`.

        ``include`` and ``exclude`` are containers of relation type
        identifiers (see :class:`RelationInfoBase`). If ``include`` is not
        ``None``, then only instances of relations in it are included in the
        result. If ``exclude`` is not ``None``, then all relations in it are
        omitted from the result. If both are ``None``, all relations are
        selected.

        If ``skip_artificial`` is ``True`` (the default), then artificial
        synsets (see :attr:`SynsetBase.is_artificial`) are "skipped over": new
        relation edges are created to replace ones ending or staring in an
        artificial synset, and connecting neighbouring synsets if they have
        relations directed like this::

            .-------.  Rel 1
            | Syn D |-----------------.
            '-------'                 |
                                      v
                              .--------------.
            .-------.  Rel 1  |    Syn B     |  Rel 1  .-------.
            | Syn A |-------->| [artificial] |-------->| Syn E |
            '-------'         '--------------'         '-------'
                                      ^
                                      |
            .-------.  Rel 2          |
            | Syn C |-----------------'
            '-------'


            .-------.  Rel 1
            | Syn D |-----------------.
            '-------'                 v
                                  .-------.
                                  | Syn E |
                                  '-------'
            .-------.  Rel 1          ^
            | Syn A |-----------------'
            '-------'

        ``Syn C`` is dropped, since there's no instance of ``Rel 1`` directed
        outwards from the skipped artificial ``Syn B``.
        """
        pass

    @abstractmethod
    def lexical_relation_edges(self, include=None, exclude=None):
        """Get an iterable of lexical unit relation instances from plWordNet.

        As represented by :class:`RelationEdge`.

        This method works like :meth:`.synset_relation_edges`, but for lexical
        units and relation types. There is no ``skip_artificial``, since there
        are no artificial lexical units.
        """
        pass

    @abstractmethod
    def relations_info(self, name=None, kind=None):
        """Get an iterable of :class:`RelationInfoBase` instances.

        Matching the query defined by parameters.

        ``name`` is a string naming a relation (see
        :class:`RelationInfoBase`). If it names a "parent", all its children
        are selected.

        ``kind`` is an enumerated value of
        :class:`~plwn.enums.RelationKind`.

        Any parameter that's not passed matches any relation type.
        As such, a call of ``relations_info()`` will select all relation types
        in plWordNet.
        """
        pass

    def close(self):
        """Perform cleanup operations.

        After using the :class:`PLWordNetBase` object.

        By default, this method does nothing and should be overridden by a
        subclass if necessary. It should still always be called, since any
        :class:`PLWordNetBase` subclass may create any kind of temporary
        resources.

        After calling this method, this instance and any ones linked with it
        (:class:`SynsetBase`, :class:`LexicalUnitBase`, etc.) may become
        invalid and should not be used.
        """
        pass

    def to_graphml(self,
                   out_file,
                   graph_type=go.GRAPH_TYPE_SYNSET,
                   include_attributes=False,
                   prefix_ids=False,
                   included_synset_attributes=None,
                   excluded_synset_attributes=None,
                   included_lexical_unit_attributes=None,
                   excluded_lexical_unit_attributes=None,
                   included_synset_relations=None,
                   excluded_synset_relations=None,
                   included_lexical_unit_relations=None,
                   excluded_lexical_unit_relations=None,
                   included_synset_nodes=None,
                   excluded_synset_nodes=None,
                   included_lexical_unit_nodes=None,
                   excluded_lexical_unit_nodes=None,
                   skip_artificial_synsets=True):
        """Export plWordNet as graph.

        In `GraphML <http://graphml.graphdrawing.org/>`_ format.

        Nodes of the graph are synsets and / or lexical units, and edges are
        relation instances.

        For nodes, their numeric plWordNet IDs are set as their XML element
        IDs.

        **NOTE:** Nodes that have no inbound or outbound edges are dropped from
        the graph.

        Nodes and edges have attributes, as GraphML defines them. For nodes,
        attributes are public properties of :class:`SynsetBase` or
        :class:`LexicalUnitBase` (aside from ``relations``, which would be
        useless in a graph, and ``id``, which becomes the XML ID of a node).
        Edges have two attributes:

        * **type**: Either ``relation``, for edges that represent plWordNet
          relation instances, or ``unit_and_synset`` for edges between synset
          nodes and nodes of lexical units that belong to the synset. The
          latter appear only in *mixed* graph.
        * **name**: If **type** is ``relation``, then this is the full name
          of the relation (see :class:`RelationInfoBase`). If **type** is
          ``unit_and_synset``, it is one of constant values: ``has_unit`` if
          the edge is directed from synset to unit, or ``in_synset``, for edges
          directed from unit to synset.

        ``out_file`` is a writable file-like object to which the GraphML output
        will be written.

        ``graph_type`` is one of three constant string values: ``synset``,
        ``lexical_unit`` or ``mixed``. Synset graph contains only synset
        nodes and relations, lexical unit graph contains only lexical unit
        nodes and relations, and mixed graph contains all of the former, as
        well as additional edges that map lexical units to synsets they belong
        to.

        If ``include_attributes`` is ``True``, then all synset and / or lexical
        unit attributes will be included. By default, attributes are not
        included to shrink the written file. Note, that if any of
        ``(included/excluded)_(synset/lexical_unit)_attributes`` parameters is
        passed, inclusion of attributes will be controlled by them and the
        value of ``include_attributes`` is ignored.

        If ``prefix_ids`` is ``True``, then ID of each node will be prefixed
        with the type: ``synset-`` or ``lexical_unit-``. By default, it's
        not done, unless ``graph_type`` is ``mixed``, in which case this
        parameter is ignored and ID prefixes are enforced.

        ``included_synset_attributes`` and ``excluded_synset_attributes`` are
        containers of synset attribute names, selecting the values which should
        or should not be included with synset nodes.

        ``included_lexical_unit_attributes`` and
        ``excluded_lexical_unit_attributes`` are the same way as the above,
        but for attributes of lexical units.

        ``included_synset_relations`` and ``excluded_synset_relations`` are
        containers of synset relation type identifiers (see
        :class:`RelationInfoBase`), selecting synset relation types whose
        instances should or should not be included in the graph. By default,
        all relation types are included.

        ``included_lexical_unit_relations`` and
        ``excluded_lexical_unit_relations`` are the same was as the above, but
        for lexical relation types.

        ``included_synset_nodes`` and ``excluded_synset_nodes`` are containers
        for IDs of synset that should or should not be included as nodes in the
        graph. If a node is not included, all edges that start or end in it are
        also excluded. By default, all non-artificial synsets are included.

        ``included_lexical_unit_nodes`` and ``excluded_lexical_unit_nodes`` are
        the same way as the above, but for lexical units.

        If ``skip_artificial_synsets`` is ``True`` (the default), then
        artificial synsets are excluded from the graph, and edges connecting to
        them are reconnected to "skip over" them, as described for
        :meth:`.synset_relation_edges`.

        **Note:** while this method accepts all of the above parameters at
        all times, parameters relating to synsets are ignored if ``graph_type``
        is ``lexical_unit``, and parameters relating to lexical units are
        ignored if ``graph_type`` is ``synset``.
        """
        gwn = go.GraphMLWordNet()
        gb = go.GraphMLBuilder(self, gwn)

        if graph_type == go.GRAPH_TYPE_SYNSET:
            gb.synset_graph(
                prefix_ids=prefix_ids,
                include_attributes=include_attributes,
                included_attributes=included_synset_attributes,
                excluded_attributes=excluded_synset_attributes,
                included_nodes=included_synset_nodes,
                excluded_nodes=excluded_synset_nodes,
                included_relations=included_synset_relations,
                excluded_relations=excluded_synset_relations,
                skip_artificial_synsets=skip_artificial_synsets,
            )
        elif graph_type == go.GRAPH_TYPE_UNIT:
            gb.lexical_unit_graph(
                prefix_ids=prefix_ids,
                include_attributes=include_attributes,
                included_attributes=included_lexical_unit_attributes,
                excluded_attributes=excluded_lexical_unit_attributes,
                included_nodes=included_lexical_unit_nodes,
                excluded_nodes=excluded_lexical_unit_nodes,
                included_relations=included_lexical_unit_relations,
                excluded_relations=excluded_lexical_unit_relations,
            )
        elif graph_type == go.GRAPH_TYPE_MIXED:
            gb.mixed_graph(
                include_attributes=include_attributes,
                included_synset_attributes=included_synset_attributes,
                excluded_synset_attributes=excluded_synset_attributes,
                included_lexical_unit_attributes=(
                    included_lexical_unit_attributes),
                excluded_lexical_unit_attributes=(
                    excluded_lexical_unit_attributes),
                included_synset_relations=included_synset_relations,
                excluded_synset_relations=excluded_synset_relations,
                included_lexical_unit_relations=(
                    included_lexical_unit_relations),
                excluded_lexical_unit_relations=(
                    excluded_lexical_unit_relations),
                included_synset_nodes=included_synset_nodes,
                excluded_synset_nodes=excluded_synset_nodes,
                included_lexical_unit_nodes=included_lexical_unit_nodes,
                excluded_lexical_unit_nodes=excluded_lexical_unit_nodes,
                skip_artificial_synsets=skip_artificial_synsets,
            )
        else:
            raise ValueError('Invalid graph type: {!r}'.format(graph_type))

        gwn.write(out_file)

    def __repr__(self):
        return '<PLWordNet ({}) at {:x}>'.format(
            self._STORAGE_NAME,
            id(self),
        )


@six.python_2_unicode_compatible
@six.add_metaclass(ABCMeta)
class SynsetBase(object):
    """Encapsulates data associated with a plWordNet synset.

    Synset contains lexical units that have the same meaning (ie. synonyms).
    Most of plWordNet relations are between meanings, hence the need to group
    lexical units into synsets.

    For purposes of ordering, a :class:`SynsetBase` object is uniquely
    identified by its "head": the first of the lexical units it contains.
    """

    @abstractproperty
    def id(self):
        """The internal, numeric identifier of the synset in plWordNet.

        It is unique among all synsets.

        If this identifier is passed to :meth:`PLWordNetBase.synset_by_id`, it
        would return this :class:`SynsetBase` object.
        """
        pass

    @abstractproperty
    def lexical_units(self):
        """Tuple of :class:`LexicalUnitBase` objects.

        Representing lexical units contained in the synset.
        Ordering of units within the tuple is arbitrary, but constant.

        At least one lexical unit is always present in every synset, so
        ``lexical_units[0]`` is always valid and selects the synset's "head".
        """
        pass

    @abstractproperty
    def definition(self):
        """Textual description of the synset's meaning.

        May be ``None``.

        In plWordNet, most definitions are stored as
        :attr:`LexicalUnitBase.definition`. Synset definitions are present
        mostly for English synsets.
        """
        pass

    @abstractproperty
    def is_artificial(self):
        """Boolean value informing if the synset is an artificial one.

        Artificial synsets carrying no linguistic
        meaning, but introduced as a method of grouping synsets within the
        structure of plWordNet.

        For most uses, artificial synsets should be ignored.
        """
        pass

    @abstractproperty
    def relations(self):
        """Tuple of :class:`RelationInfoBase` instances.

        Containing types of distinct relations that have outbound
        edges from this synset.

        Relations are returned in an arbitrary order.

        The tuple is special: methods for checking membership accept all
        possible representations of a relation type (see
        :meth:`RelationInfoBase.eqv`).
        """
        pass

    @abstractproperty
    def is_polish(self):
        """Check whether all units are Polish."""
        pass

    @abstractproperty
    def is_english(self):
        """Check whether all units are English."""
        pass

    @abstractproperty
    def pos(self):
        """Returns PoS of the synset units.

        Raises :exc:`ValueError` if units have many different PoS.
        """
        pass

    @abstractmethod
    def related(self, relation_id=None, skip_artificial=True):
        """Get an iterable of :class:`SynsetBase` instances.

        That are connected to this synset by outbound edges of
        synset relation type identified by ``relation_id``.

        ``relation_id`` can be any synset relation type identifier (see
        :class:`RelationInfoBase`), a collection of relation types identifiers,
        or ``None``, in which case synsets related to this one by any relation
        are selected.

        Note, that distinction between any relations that fit the
        ``relation_id`` query is lost. Use :meth:`.related_pairs` if it's
        needed.

        Raises :exc:`~plwn.exceptions.InvalidRelationTypeException` if
        (any of) ``relation_id`` does not refer to an existing synset relation
        type.

        If ``skip_artificial`` is ``True`` (the default) artificial synsets
        related to this one are "skipped over", as described for
        :meth:`PLWordNetBase.synset_relation_edges`.
        """
        pass

    @abstractmethod
    def related_pairs(self, relation_id=None, skip_artificial=True):
        """Like :meth:`.related`.

        But return an iterable of pairs
        ``(<relation info>, <relation target synset>)``.
        """
        pass

    def to_dict(self, include_related=True, include_units_data=True):
        """Create a JSON-compatible dictionary.

        With all public properties of the synset.

        Enums are converted to their values and all collections are converted
        to tuples.

        Property :attr:`.relations` is omitted, as it would be redundant when
        all related synsets can be enumerated when ``include_related`` is
        ``True``. Some additional members are also present in the dictionary:

        * ``str``: The string representation of the synset (defined by
          ``__str__`` override on :class:`SynsetBase`).
        * ``units``: Listing (as a tuple) of units belonging to the synset (in
          the same ordering as :attr:`.lexical_units`), as pairs of
          ``(<unit id>, <unit string form>)``.

        If ``include_related`` is ``True`` (the default), the dictionary will
        contain an additional ``related`` member, representing synsets related
        to this one, in the following format::

            {
                <synset relation full name>: (
                    (<relation target id>, <relation target string form>),
                    ...
                ),
                ...
            }

        If ``include_units_data`` is ``True`` (the default), the ``units``
        member will contain results of invocation of
        :meth:`LexicalUnitBase.to_dict` for the synset's units,
        instead of pairs described above. In this case, the value of
        ``include_related`` parameter is passed on to
        :meth:`LexicalUnitBase.to_dict`.
        """
        syn_dict = {
            u'id': self.uuid,
            u'definition': self.definition,
            u'is_artificial': self.is_artificial,
            u'units': tuple(
                (lu.to_dict(include_related) for lu in self.lexical_units)
                if include_units_data
                else ((lu.id, six.text_type(lu)) for lu in self.lexical_units)
            ),
            u'str': six.text_type(self),
        }

        if include_related:
            syn_dict[u'related'] = {
                six.text_type(rel): tuple(
                    (target.uuid, target.short_str())
                    for target in self.related(rel)
                )
                for rel in self.relations
            }

        return syn_dict

    def short_str(self):
        """Shorter version of synset's string form (``__str__``).

        That displays only the first lexical unit.
        """
        sstr = [u'{', six.text_type(self.lexical_units[0])]
        if len(self.lexical_units) > 1:
            sstr.append(
                u', [+ {} unit(s)]'.format(len(self.lexical_units) - 1),
            )
        sstr.append(u'}')
        return ''.join(sstr)

    def __inner_cmp(self, cmp_op, other):
        if not isinstance(other, SynsetBase):
            return NotImplemented
        return cmp_op(self.lexical_units[0], self.lexical_units[0])

    def __repr__(self):
        head = self.lexical_units[0]
        rstr = '<Synset id={!r} lemma={!r} pos={!r} variant={!r}'.format(
            str(self.uuid),
            head.lemma,
            head.pos,
            head.variant,
        )

        if len(self.lexical_units) > 1:
            rstr += ' [+ {} unit(s)]'.format(len(self.lexical_units) - 1)

        return rstr + '>'

    def __str__(self):
        return (
            u'{' +
            u', '.join(six.text_type(lu) for lu in self.lexical_units) +
            u'}'
        )

    def __hash__(self):
        # Even if comparing is done by the synset's head, it's probably better
        # to hash by all lexical units, to boost the hash's uniqueness
        return hash((SynsetBase, self.lexical_units))

    def __eq__(self, other):
        return self.__inner_cmp(op.eq, other)

    def __ne__(self, other):
        return self.__inner_cmp(op.ne, other)

    def __lt__(self, other):
        return self.__inner_cmp(op.lt, other)

    def __le__(self, other):
        return self.__inner_cmp(op.le, other)

    def __gt__(self, other):
        return self.__inner_cmp(op.gt, other)

    def __ge__(self, other):
        return self.__inner_cmp(op.ge, other)


@six.python_2_unicode_compatible
@six.add_metaclass(ABCMeta)
class LexicalUnitBase(object):
    """Encapsulates data associated with a plWordNet lexical unit.

    Lexical units represent terms in the language. Each lexical unit is
    uniquely identified by its lemma (base written form), part of speech
    (verb, noun, adjective or adverb) and variant (a number differentiating
    between homonyms).
    """

    @abstractproperty
    def id(self):
        """The internal, numeric identifier of the lexical units in plWordNet.

        It is unique among all lexical units.

        If this identifier is passed to
        :meth:`PLWordNetBase.lexical_unit_by_id`, it would return this
        :class:`LexicalUnitBase` object.
        """
        pass

    @abstractproperty
    def lemma(self):
        """Lemma of the unit; its basic text form."""
        pass

    @abstractproperty
    def pos(self):
        """Part of speech of the unit.

        One of enumerated constants of :class:`~plwn.enums.PoS`.
        """
        pass

    @abstractproperty
    def variant(self):
        """Ordinal number to differentiate between meanings of homonyms.

        Numbering starts at 1.
        """
        pass

    @abstractproperty
    def definition(self):
        """Textual description of the lexical unit's meaning.

        May be ``None``.
        """
        pass

    @abstractproperty
    def sense_examples(self):
        """Text fragments.

        That show how the lexical unit is used in the language.

        May be an empty tuple.
        """
        pass

    @abstractproperty
    def sense_examples_sources(self):
        """Symbolic representations of sources.

        From which the sense examples were taken.

        The symbols are short strings, defined by plWordNet.

        This tuples has the same length as :attr:`.sense_examples`, and is
        aligned by index (for example, the source of ``sense_examples[3]`` is
        at ``sense_examples_sources[3]``).

        To get pairs of examples with their sources, use
        ``zip(sense_examples, sense_examples_sources)``
        """
        # TODO List of source symbols, link to?
        pass

    @abstractproperty
    def external_links(self):
        """URLs linking to web pages describing the meaning of the lexical unit.

        May be an empty collection.
        """
        pass

    @abstractproperty
    def usage_notes(self):
        """Symbols.

        Denoting certain properties of how the lexical unit is used in
        the language.

        The symbols are short strings, defined by plWordNet. For example,
        ``daw.`` means that the word is considered dated.

        May be an empty collection.
        """
        pass

    @abstractproperty
    def domain(self):
        """plWordNet domain the lexical unit belongs to.

        One of enumerated constants of :class:`~plwn.enums.Domain`.
        """
        pass

    @abstractproperty
    def verb_aspect(self):
        """Aspect of a verb.

        Of the enumerated values of :class:`~plwn.enums.VerbAspect`.

        May be ``None`` if the unit is not a verb, or had no aspect assigned.
        """
        pass

    @abstractproperty
    def is_emotional(self):
        """Boolean value informing if the lexical unit has emotional affinity.

        If it is ``True``, then the lexical unit describes a term that has an
        emotional load, and ``emotion_*`` properties will have meaningful
        values, describing the affinity.

        If it is ``False``, then the unit is emotionally neutral. All
        ``emotion_*`` properties will be ``None`` or empty collections.

        This property can also be ``None``, which means that the unit has not
        (yet) been evaluated with regards to emotional affinity. All
        ``emotion_*`` properties are the same as when it's ``False``.
        """
        pass

    @abstractproperty
    def emotion_markedness(self):
        """Markedness of emotions associated with the lexical unit.

        May be ``None`` if the unit has no emotional markedness.

        If this property is ``None`` then all other ``emotion_*`` properties
        will be ``None`` or empty collections.
        """
        pass

    @abstractproperty
    def emotion_names(self):
        """Tuple of names of emotions associated with this lexical unit."""
        pass

    @abstractproperty
    def emotion_valuations(self):
        """Tuple of valuations of emotions associated with this lexical unit."""
        pass

    @abstractproperty
    def emotion_example(self):
        """Example of an emotionally charged sentence using the lexical unit."""
        pass

    @abstractproperty
    def emotion_example_secondary(self):
        """``Optional[str]``.

        This property is not ``None`` only if :attr:`.emotion_markedness` is
        :attr:`~plwn.enums.EmotionMarkedness.amb`. In such case,
        :attr:`.emotion_example` will be an example of a positively charged
        sentence, and this one will be a negatively charged sentence.
        """
        pass

    @abstractproperty
    def synset(self):
        """An instance of :class:`SynsetBase`.

        Representing the synset this unit belongs to.
        """
        pass

    @abstractproperty
    def relations(self):
        """Tuple of :class:`RelationInfoBase` instances.

        Containing types of distinct relations that have
        outbound edges from this lexical unit.

        Relations are returned in an arbitrary order.

        The tuple is special: methods for checking membership accept all
        possible representations of a relation type (see
        :meth:`RelationInfoBase.eqv`).
        """
        pass

    @abstractproperty
    def is_polish(self):
        """Check whether unit is Polish by its PoS."""
        pass

    @abstractproperty
    def is_english(self):
        """Check whether unit is English by its PoS."""
        pass

    @abstractmethod
    def related(self, relation_id=None):
        """Get an iterable of :class:`LexicalUnitBase` instances.

        That are connected to this lexical unit by outbound edges
        of lexical relation type identified by ``relation_id``.

        ``relation_id`` can be any lexical relation type identifier (see
        :class:`RelationInfoBase`), a collection of relation types identifiers,
        or ``None``, in which case lexical units related to this one by any
        relation are selected.

        Note, that distinction between any relations that fit the
        ``relation_id`` query is lost. Use :meth:`.related_pairs` if it's
        needed.

        Raises :exc:`~plwn.exceptions.InvalidRelationTypeException` if
        ``relation_id`` does not refer to an existing lexical relation type.
        """
        pass

    @abstractmethod
    def related_pairs(self, relation_id):
        """Like :meth:`.related`.

        But return an iterable of pairs
        ``(<relation info>, <relation target unit>)``.
        """
        pass

    def to_dict(self, include_related=True):
        """Create a JSON-compatible dictionary.

        With all the public properties of the lexical unit.

        Enums are converted to their values and all collections are converted
        to tuples.

        Property :attr:`.relations` is omitted, as it would be redundant when
        all related lexical units can be enumerated when ``include_related``
        is ``True``.

        An additional ``str`` member is present in the dictionary; its value is
        the string representation of the lexical unit.

        If ``include_related`` is ``True`` (the default), the dictionary will
        contain an additional ``related`` member, representing lexical units
        related to this one, in the following format::

            {
                <lexical relation full name>: (
                    (<relation target id>, <relation target string form>),
                    ...
                ),
                ...
            }
        """
        lu_dict = {
            u'id': self.uuid,
            u'legacy_id': self.legacy_id,
            u'lemma': self.lemma,
            u'pos': self.pos.value,
            u'variant': self.variant,
            u'definition': self.definition,
            u'sense_examples': tuple(self.sense_examples),
            u'sense_examples_sources': tuple(self.sense_examples_sources),
            u'external_links': tuple(self.external_links),
            u'usage_notes': tuple(self.usage_notes),
            u'domain': self.domain.value,
            u'synset': self.synset.uuid,
            u'verb_aspect': None
            if self.verb_aspect is None
            else self.verb_aspect.value,
            u'emotion_markedness': None
            if self.emotion_markedness is None
            else self.emotion_markedness.value,
            u'emotion_names': make_values_tuple(self.emotion_names),
            u'emotion_valuations': make_values_tuple(self.emotion_valuations),
            u'emotion_example': self.emotion_example,
            u'emotion_example_secondary': self.emotion_example_secondary,
            u'str': six.text_type(self),
        }

        if include_related:
            lu_dict[u'related'] = {
                six.text_type(rel): tuple(
                    (target.uuid, six.text_type(target))
                    for target in self.related(rel)
                )
                for rel in self.relations
            }

        return lu_dict

    def __lt_lempos(self, other):
        # Common code for __lt__ and __le__ methods.
        # Compares first two elements.
        colled = locale.strcoll(self.lemma, other.lemma)
        if colled < 0:
            return True
        if colled > 0:
            return False
        if self.pos is other.pos:
            # Defer comparison
            return None
        return self.pos.value < other.pos.value

    def __inner_eq(self, other):
        return (locale.strcoll(self.lemma, other.lemma) == 0 and
                self.pos == other.pos and
                self.variant == other.variant)

    def __inner_cmp(self, cmp_op, other):
        if not isinstance(other, LexicalUnitBase):
            return NotImplemented
        cmp_val = self.__lt_lempos(other)
        return (cmp_val
                if cmp_val is not None
                else cmp_op(self.variant, other.variant))

    def __repr__(self):
        return '<LexicalUnit id={!r} lemma={!r} pos={!r} variant={!r}>'.format(
            str(self.uuid),
            self.lemma,
            self.pos,
            self.variant,
        )

    def __str__(self):
        return u'{lem}.{var}({domnum}:{domname})'.format(
            lem=self.lemma.replace(u' ', u'_'),
            var=self.variant,
            domnum=self.domain.db_number,
            domname=self.domain.name,
        )

    def __hash__(self):
        return hash((LexicalUnitBase, self.lemma, self.pos, self.variant))

    def __eq__(self, other):
        if not isinstance(other, LexicalUnitBase):
            return NotImplemented
        return self.__inner_eq(other)

    def __ne__(self, other):
        if not isinstance(other, LexicalUnitBase):
            return NotImplemented
        return not self.__inner_eq(other)

    def __lt__(self, other):
        return self.__inner_cmp(op.lt, other)

    def __le__(self, other):
        return self.__inner_cmp(op.le, other)

    def __gt__(self, other):
        return self.__inner_cmp(op.gt, other)

    def __ge__(self, other):
        return self.__inner_cmp(op.ge, other)


@six.python_2_unicode_compatible
@six.add_metaclass(ABCMeta)
class RelationInfoBase(object):
    """Encapsulates information associated with a relation type.

    The primary purpose of this class is to serve as a single object
    consolidating all possible ways a relation type can be referred to.

    In general, plWordNet uses *parent* and *child* relation names. Child
    relations are those that have actual instances between synsets and lexical
    units. Parent relations only exist to group child relations together; child
    relation names need to be only unique within the group of their parent
    relation, while parent relations must be globally unique.

    For example, there are two relations named "część" ("part"); one being a
    child of "meronimia" ("meronymy"), and another a child of "holonimia"
    ("holonymy").

    Some relation types have no parent; they behave like child relations, but
    their names need to be unique on par with parent relations.

    plWordNet also stores shorter aliases for most of the relation types,
    for example "hipo" for "hiponimia" ("hyponymy").

    There are four ways to refer to relations wherever a relation identifier
    is accepted (usually the argument is named ``relation_id``):

    * Full name, in format ``<parent name>/<child name>`` (or just
      ``<child name>`` if the relation has no parent).
    * One of the shorter aliases mentioned above. This is checked before
      attempting to resolve relation names. Aliases must be globally unique.
    * A parent name on its own. This resolves to all children of the parent
      relation. Note, that it's not always valid to pass a name that resolves
      to multiple relations;
      :exc:`~plwn.exceptions.AmbiguousRelationTypeException` is raised in such
      cases.
    * Finally, a :class:`RelationInfoBase` instance may be used instead of
      a string, standing for the child relation it represents.

    Note, that parent relations don't have corresponding
    :class:`RelationInfoBase` instance.
    """

    #: Character that separates parent from child name in full name
    #: representation. It must not appear in any relation names or aliases.
    SEP = u'/'

    @classmethod
    def format_name(cls, parent_name, child_name):
        """Format and return a full name out of parent and child name strings.

        ``parent_name`` may be ``None``, which will just return ``child_name``,
        as relations without parents are fully represented just by their name.
        """
        parform = u'' if parent_name is None else parent_name + cls.SEP
        return parform + child_name

    @classmethod
    def split_name(cls, full_name):
        """Split a full name into a ``(<parent name>, <child name>)`` pair.

        ``parent_name`` may be ``None`` if :attr:`.SEP` doesn't appear in the
        full name.

        However, if :attr:`.SEP` appears more than once in ``full_name``, a
        ``ValueError`` will be raised.
        """
        items = full_name.split(cls.SEP)
        itlen = len(items)

        if itlen > 2:
            raise ValueError(full_name)

        return (None, items[0]) if itlen < 2 else tuple(items)

    @abstractproperty
    def kind(self):
        """One of enumerated constants of :class:`~plwn.enums.RelationKind`.

        Denotes it's a synset or lexical relation.
        """
        pass

    @abstractproperty
    def parent(self):
        """String name of the parent relation to this one.

        May be ``None`` if the relation has no parent.
        """
        pass

    @abstractproperty
    def name(self):
        """String name of the relation."""

    @abstractproperty
    def aliases(self):
        """Tuple of all aliases the relation can be referred to by."""
        pass

    def eqv(self, other):
        """Check if ``other`` is an equivalent representation.

        Either an equal :class:`RelationInfoBase` object or
        a relation identifier that refers to this object.

        This is less strict than the equality operator, which only checks for
        equal :class:`RelationInfoBase` instances.
        """
        sother = six.text_type(other)
        return sother == six.text_type(self) or sother in self.aliases

    def __inner_eq(self, other):
        return (self.parent == other.parent and
                self.name == other.name)

    def __inner_cmp(self, cmp_op, other):
        if not isinstance(other, RelationInfoBase):
            return NotImplemented
        return cmp_op(six.text_type(self), six.text_type(other))

    def __repr__(self):
        return (
            '<RelationInfo name={!r} parent={!r} kind={!r} aliases={!r}>'
            .format(
                self.name,
                self.parent,
                self.kind,
                self.aliases,
            )
        )

    def __str__(self):
        return self.format_name(self.parent, self.name)

    def __hash__(self):
        return hash((RelationInfoBase, self.parent, self.name))

    def __eq__(self, other):
        if not isinstance(other, RelationInfoBase):
            return NotImplemented
        return self.__inner_eq(other)

    def __ne__(self, other):
        if not isinstance(other, RelationInfoBase):
            return NotImplemented
        return not self.__inner_eq(other)

    def __lt__(self, other):
        return self.__inner_cmp(op.lt, other)

    def __le__(self, other):
        return self.__inner_cmp(op.le, other)

    def __gt__(self, other):
        return self.__inner_cmp(op.gt, other)

    def __ge__(self, other):
        return self.__inner_cmp(op.ge, other)
