import InfoTooltip from "./InfoTooltip.js";
import Spinner from "./Spinner.js";
import UnificationSwitchableList from "./UnificationSwitchableList.js";
import ExamplesComponent from "./frame-components/ExamplesComponent.js";
import SlowalFrameComponent from "./frame-components/SlowalFrameComponent.js";
import SemanticsSchemataComponent from "./frame-components/SemanticsSchemataComponent.js";

let LexicalUnitEdit = {};

const FramePreview = {
  props: {
    initialUnifiedFrameId: Number,
    initialLexicalUnitId: Number,
    forceRefresh: Number
  },
  data () {
    return {
      unifiedFrameId: this.initialUnifiedFrameId
    };
  },
  components: {LexicalUnitEdit, UnificationSwitchableList},
  emits: ['changeFrame', 'refreshEntriesList'],
  methods: {
    unifiedFrameSelected (unifiedFrameId) {
      this.$emit('changeFrame', unifiedFrameId);
    },
    refreshEntriesList() {
      this.$emit('refreshEntriesList')
    },
  },
  mounted () {
    Split(['#frame-preview-left-pane', '#frame-preview-right-pane'], {
      sizes: [60, 40],
      minSize: 10,
      gutterSize: 4,
      elementStyle: (dimension, size, gutterSize) => {
        return {
          'flex-basis': 'calc(' + size + '% - ' + gutterSize + 'px)'
        }
      },
    });
  },
  template: `
    <div class="row h-100 overflow-hidden">
      <div :key="unifiedFrameId" class="col h-100 pr-0 pt-0 pb-0 overflow-auto" id="frame-preview-left-pane">
        <lexical-unit-edit
          v-if="unifiedFrameId !== -1"
          :readOnly="true"
          :unifiedFrameId="unifiedFrameId"
          :forceRefresh="forceRefresh"
          @refresh-entries-list="refreshEntriesList"
        />
        <div v-else class="h-100">
          Brak ramy do wyświetlenia
        </div>
      </div>
      <div class="col h-100 pl-1 pt-0 pb-0 overflow-auto" id="frame-preview-right-pane">
        <unification-switchable-list
          @unified-frame-selected="unifiedFrameSelected"
        />
      </div>
    </div>
  `
};

Object.assign(LexicalUnitEdit, {
  props: {
    unifiedFrameId: Number,
    previewedUnifiedFrameId: Number,
    readOnly: Boolean,
    initialRightPaneTab: String,
    forceRefresh: Number,
  },
  data() {
    return {
      gettext: window.gettext,
      unified_frame: {},
      unified_frame_title: '',
      unified_frame_arguments: [],
      active_unified_frame_argument: null,
      slowal_frames2selecional_preferencies_mapping: {},
      lexical_units: [],
      img_prefix: String,
      frames: [],
      right_pane_tabs: [
        {id: 'schemata', label: gettext('Schematy')},
        {id: 'frame_preview', label: gettext('Podgląd ram')},
        {id: 'notes', label: gettext('Notatki')},
      ],
      right_pane_tab: this.initialRightPaneTab || 'schemata',
      currentPreviewedUnifiedFrameId: this.previewedUnifiedFrameId,
      internalForceRefresh: this.forceRefresh,
      statusButtonTitle: '',
      active_slowal_frame: null,
      showVerifiedFrames: false,
      subentries: null,
      alternations: null,
      realisation_phrases: null,
      realisation_descriptions: null,
      examples: null,
      selectedFrameArguments: null,
      selectedLus: null,
      selectedSchemas: null,
      hidden_frames: [],
    }
  },
  components: {InfoTooltip, Spinner, FramePreview, SlowalFrameComponent, ExamplesComponent, SemanticsSchemataComponent},
  emits: ['goToDisplay', 'refresh', 'swapFrames', 'refreshEntriesList', 'clearUnifiedFrameView'],
  watch: {
    forceRefresh(newVal, oldVal) {
      this.loadFrame();
    }
  },
  methods: {
    hasWhiteSpace(s) {
      return /\s/g.test(s);
    },
    slowal_frame_html(frame) {
      let ret = frame2dom(frame);
      return ret.outerHTML;
    },
    async loadFrame() {
      try {
        const data = {'unified_frame_id': this.unifiedFrameId};
        $.ajax({
          type: 'post',
          url: '/' + lang + '/unifier/get_unified_frame/',
          dataType: 'json',
          data: data,
          timeout: 60000,
          success: function (response) {

            this.img_prefix = window.STATIC_URL;
            this.lexical_units = frames2lexical_units(response.frames);
            this.unified_frame = response.unified_frame;
            this.unified_frame_title = this.unified_frame.title;
            this.unified_frame_arguments = this.unified_frame.arguments;
            this.frames = response.frames;
            this.slowal_frames2selecional_preferencies_mapping = slowal_frames2selecional_preferencies(this.unified_frame, response.frames);

            this.subentries = response.subentries;
            this.alternations = response.alternations;
            this.realisation_phrases = response.realisation_phrases;
            this.realisation_descriptions = response.realisation_descriptions;
            this.examples = response.examples;

            window.fulfill_slowal_frames_arguments_with_empty_elems(response.unified_frame, response.frames)
            window.update_last_visited(response.last_visited);
            window.clear_info();

            this.changeStatusButtonTitleToDefault();
            if(!this.active_slowal_frame) {
              this.setup_notes_unified_frame();
            }
          }.bind(this),
          error: function (request, errorType, errorMessage) {
            show_error(errorType + ' (' + errorMessage + ')');
          }
        });

      } catch (error) {
        console.log(error);
      }
    },
    setup_notes_unified_frame () {
      setup_notes($('#notes-component'), $('#lexical-unit-notes-template'), this.unified_frame.id, 'unifier.UnifiedFrame', this.setup_notes_unified_frame);
    },
    setup_notes_slowal_frame () {
      if(this.active_slowal_frame) {
        setup_notes($('#notes-component'), $('#lexical-unit-notes-template'), this.active_slowal_frame.id, 'semantics.Frame', this.setup_notes_slowal_frame);
      }
    },
    setup_notes_slowal_frame_with_title_and_body (title, body) {
      this.right_pane_tab = 'notes';
      setup_notes($('#notes-component'), $('#lexical-unit-notes-template'), this.active_slowal_frame.id, 'semantics.Frame', this.setup_notes_slowal_frame, title, body);
    },
    unifiedFrameArgumentSelected(argument) {
      if (this.active_unified_frame_argument === argument) {
        this.active_unified_frame_argument = null;
      } else {
        this.active_unified_frame_argument = argument;
      }
      this.unifiedFrameArgumentHovered(argument);
    },
    unifiedFrameArgumentHovered(argument) {
      clear_info();
      if (argument && this.active_unified_frame_argument === argument) {
        show_info(gettext('Kliknij, aby cofnąć wybór kolumny do edycji.'));
      }
      if (argument && this.active_unified_frame_argument !== argument) {
        show_info(gettext('Kliknij, aby wybrać kolumnę do edycji.'));
      }
    },
    addSelectivePreference () {
      if (!this.active_unified_frame_argument) {
        alert(gettext("Zaznacz argument, do którego chcesz dodać preferencję."));
      } else {
        window.addSelectivePreference(this.unified_frame, this.active_unified_frame_argument.id, this.frames, function () {
          this.loadFrame();
        }.bind(this));
      }
    },
    removeSelectionalPreference () {
      if (!this.active_unified_frame_argument) {
        alert(gettext("Zaznacz argument, do którego chcesz dodać preferencję."));
      } else {
        const existingPreferencies = function () {
          return this.active_unified_frame_argument.preferences.map(preference => {
            return `<label><input type="checkbox" name="preference" value="${preference.id}" /> ${preference.str}</label><br />`;
          }).join("");
        }.bind(this);

        const remove_preference_popup = {
          state0: {
            title: 'Wybierz preferencje do usunięcia',
            html: existingPreferencies,
            buttons: {Anuluj: 0, Usuń: 1},
            focus: -1,
            submit: function (e, v, m, f) {
              if (v == 0) {
                e.preventDefault();
                $.prompt.close();
              }
              if (v === 1) {
                e.preventDefault();
                let preference_ids = normalizeFormData(f.preference);
                const data = {
                  'unified_frame_id': this.unified_frame.id,
                  'argument_id': this.active_unified_frame_argument.id,
                  'preference_ids': JSON.stringify(preference_ids)
                };
                $.ajax({
                  type: 'post',
                  url: '/' + lang + '/unifier/remove_selectional_preference/',
                  dataType: 'json',
                  data: data,
                  timeout: 60000,
                  success: function (response) {
                    show_info('Wybrane preferencje zostału usunięte.');
                    this.loadFrame();
                    $.prompt.close();
                  }.bind(this),
                  error: function (request, errorType, errorMessage) {
                    show_error(errorType + ' (' + errorMessage + ')');
                    $.prompt.close();
                  }
                });
              }
            }.bind(this)
          }
        }
        $.prompt(remove_preference_popup);
      }
    },
    changeTitle() {
      let title = this.unified_frame.title != null ? this.unified_frame.title : '';
      const change_title_popup = {
        state0: {
          title: 'Zmiana nazwy ramy',
          html: '<input type="text" size="32" value="' + title + '" name="title" />',
          buttons: {Anuluj: 0, Zapisz: 1},
          focus: -1,
          submit: function (e, v, m, f) {
            if (v == 0) {
              $.prompt.close();
            }
            if (v === 1) {
              e.preventDefault();
              const title = f.title;

              if (this.hasWhiteSpace(title)) {
                alert(gettext("Nazwa zunifikowanej ramy nie może zawierać białych znaków."));
              } else {
                const data = {'unified_frame_id': this.unified_frame.id, 'unified_frame_title': title};
                $.ajax({
                  type: 'post',
                  url: '/' + lang + '/unifier/save_unified_frame_title/',
                  dataType: 'json',
                  data: data,
                  timeout: 60000,
                  success: function (response) {
                    show_info('Tytuł ramy zosał zapisany');
                    $.prompt.close();
                    this.loadFrame();
                  }.bind(this),
                  error: function (request, errorType, errorMessage) {
                    show_error(errorType + ' (' + errorMessage + ')');
                    $.prompt.close();
                  }
                });
              }
            }
          }.bind(this)
        }
      }
      $.prompt(change_title_popup);
    },
    changeRole() {
      if (!this.active_unified_frame_argument) {
        alert(gettext("Zaznacz argument, dla którego chcesz wybrać rolę."));
      } else {
        const existingSelect = function () {
          let selected_unified_frame_argument = this.unified_frame_arguments.find(o => o.id === this.active_unified_frame_argument.id);
          if (!selected_unified_frame_argument.proposed_roles) {
            return gettext('Brak ról do wyboru.')
          }
          return selected_unified_frame_argument.proposed_roles.map(role => {
            return `<label><input type="radio" name="roles" value="${role.id}" /> ${role.str}</label><br />`;
          }).join("");
        }.bind(this);

        const newSelect = function () {
          let rolesHTML = roles.map(role => {
            return `<label><input type="radio" name="role" value="${role.id}" /> ${role.role}</label><br />`;
          }).join("");
          let attributesHTML = role_attributes.map(attribute => {
            return `<label><input type="radio" name="attribute" value="${attribute.id}" /> ${attribute.attribute}</label><br />`;
          }).join("");
          const roleTypeHTML = ['role', 'modifier'].map(type => {
            return `<label><input type="radio" name="role_type" value="${type}" /> ${type}</label><br />`;
          }).join("");
          return '<div class="row">' +
              '<div class="column"><div class="role_select_header">Type</div>' + roleTypeHTML + '</div>' +
              '<div class="column"><div class="role_select_header">Role</div>' + rolesHTML + '</div>' +
              '<div class="column"><div class="role_select_header">Atrybuty</div>' + attributesHTML + '</div>' +
              '</div>';
        }.bind(this);

        let change_role_popup = {
          state0: {
            title: 'Wybór roli',
            html: 'Wybierz lub dodaj rolę',
            buttons: {Wybierz: 0, Dodaj: 1, Koniec: -1},
            focus: -1,
            submit: function (e, v, m, f) {
              if (v == -1) {
                e.preventDefault();

                this.loadFrame();

                $.prompt.close();
              }
              if (v === 0) {
                e.preventDefault();
                $.prompt.goToState('state1');
              }
              if (v == 1) {
                e.preventDefault();
                $.prompt.goToState('state2');
              }
            }.bind(this)
          },
          state1: {
            title: 'Wybierz rolę',
            html: existingSelect(),
            buttons: {Anuluj: -1, Zatwierdź: 1},
            focus: -1,
            submit: function (e, v, m, f) {
              if (v == -1) {
                e.preventDefault();
                $.prompt.goToState('state0');
              }
              if (v == 1) {
                e.preventDefault();
                normalizeFormData(f.roles).map(role_id => {
                  const data = {
                    'unified_frame_id': this.unified_frame.id,
                    'complement_id': this.active_unified_frame_argument.id,
                    'role_id': role_id
                  };
                  $.ajax({
                    type: 'post',
                    url: '/' + lang + '/unifier/save_selected_role/',
                    dataType: 'json',
                    data: data,
                    timeout: 60000,
                    success: function (response) {
                      show_info('Wybrana rola zosała zapisana');
                      $.prompt.goToState('state0');
                    }.bind(this),
                    error: function (request, errorType, errorMessage) {
                      show_error(errorType + ' (' + errorMessage + ')');
                      $.prompt.close();
                    }
                  });
                });
              }
            }.bind(this)
          },
          state2: {
            title: 'Dodaj rolę',
            html: newSelect(),
            buttons: {Anuluj: -1, Zatwierdź: 1},
            focus: -1,
            submit: function (e, v, m, f) {
              if (v == -1) {
                e.preventDefault();
                $.prompt.goToState('state0');
              }
              if (v == 1) {
                e.preventDefault();

                const role_id = normalizeFormData(f.role)[0];
                const role_type = normalizeFormData(f.role_type)[0];

                if (role_id != null && role_type != null) {
                  const attribute_id = normalizeFormData(f.attribute)[0];

                  const data = {
                    'unified_frame_id': this.unified_frame.id,
                    'complement_id': this.active_unified_frame_argument.id,
                    'role_type': role_type,
                    'role_id': role_id,
                    'attribute_id': attribute_id
                  };
                  $.ajax({
                    type: 'post',
                    url: '/' + lang + '/unifier/save_new_role/',
                    dataType: 'json',
                    data: data,
                    timeout: 60000,
                    success: function (response) {
                      show_info('Nowa rola zosała zapisana');
                      $.prompt.goToState('state0');
                    }.bind(this),
                    error: function (request, errorType, errorMessage) {
                      show_error(errorType + ' (' + errorMessage + ')');
                      $.prompt.close();
                    }
                  });
                } else {
                  alert(gettext("Musisz wybrać typ oraz rolę."));
                }
              }
            }.bind(this)
          }
        };
        $.prompt(change_role_popup);
      }
    },
    addArgument() {
      const data = {'unified_frame_id': this.unified_frame.id};
      $.ajax({
        type: 'post',
        url: '/' + lang + '/unifier/add_argument/',
        dataType: 'json',
        data: data,
        timeout: 60000,
        success: function (response) {
          show_info('Nowy argument zosał dodany');
          this.loadFrame();
        }.bind(this),
        error: function (request, errorType, errorMessage) {
          show_error(errorType + ' (' + errorMessage + ')');
          $.prompt.close();
        }
      });
    },
    removeArgument() {
      if (!this.active_unified_frame_argument) {
        alert(gettext("Zaznacz argument, który chcesz usunąć."));
      } else {
        let hasSlowalFrameArgumentMapping = false;
        for (let i in this.unified_frame.slowal_frame_mapping) {
          const slowal_frame_mapping = this.unified_frame.slowal_frame_mapping[i];
          for (let j in slowal_frame_mapping.slowal_frame_argument_mapping) {
            const slowal_frame_argument_mapping = slowal_frame_mapping.slowal_frame_argument_mapping[j];
            if (slowal_frame_argument_mapping.unified_frame_agrument_id == this.active_unified_frame_argument.id) {
              hasSlowalFrameArgumentMapping = true;
              break;
            }
          }
        }
        if (hasSlowalFrameArgumentMapping) {
          alert(gettext("Zaznaczony argument nie może zostać usunięty - podpięte ramy posiadają do niego dowiązania."));
        } else {
          const data = {
            'unified_frame_id': this.unified_frame.id,
            'complement_id': this.active_unified_frame_argument.id
          };
          $.ajax({
            type: 'post',
            url: '/' + lang + '/unifier/remove_argument/',
            dataType: 'json',
            data: data,
            timeout: 60000,
            success: function (response) {
              show_info('Wybrany argument zosał usunięty');
              this.loadFrame();
            }.bind(this),
            error: function (request, errorType, errorMessage) {
              show_error(errorType + ' (' + errorMessage + ')');
              $.prompt.close();
            }
          });
        }
      }
    },
    duplicate() {
      let title = this.unified_frame.title != null ? this.unified_frame.title : '';
      const duplicate_popup = {
        state0: {
          title: 'Podaj nazwę zduplikowanej ramy',
          html: '<input type="text" size="32" value="KOPIA_' + title + '" name="title" />',
          buttons: {Anuluj: 0, Zapisz: 1},
          focus: -1,
          submit: function (e, v, m, f) {
            if (v == 0) {
              e.preventDefault();
              $.prompt.close();
            }
            if (v === 1) {
              e.preventDefault();
              const title = f.title;

              if (this.hasWhiteSpace(title)) {
                alert(gettext("Nazwa zunifikowanej ramy nie może zawierać białych znaków."));
              } else {
                const data = {
                  'unified_frame_id': this.unified_frame.id,
                  'target_unified_frame_title': title
                };
                $.ajax({
                  type: 'post',
                  url: '/' + lang + '/unifier/duplicate_unified_frame/',
                  dataType: 'json',
                  data: data,
                  timeout: 60000,
                  success: function (response) {
                    show_info('Zunifikowana rama została zduplikowana.');
                    this.currentPreviewedUnifiedFrameId = response.unified_frame_id;
                    $.prompt.close();
                  }.bind(this),
                  error: function (request, errorType, errorMessage) {
                    show_error(errorType + ' (' + errorMessage + ')');
                    $.prompt.close();
                  }
                });
              }
            }
          }.bind(this)
        }
      }
      $.prompt(duplicate_popup);
    },
    changeUnifiedFrameStatusToReady() {
      let foundNotVerifiedFrame = this.frames.find(frame => frame.status !== 'G' && frame.status !== 'S');
      if (foundNotVerifiedFrame) {
        alert(gettext("Wszystkie podpięte ramy powinny być zweryfikowane."));
      } else {
        let roleDict = {};
        let hasPreferenceSelected = true;
        for (let i in this.unified_frame_arguments) {
          const argument = this.unified_frame_arguments[i];
          if(argument.role) {
            roleDict[argument.role.str] = argument.role;
          }
          if(!argument.preferences || argument.preferences.length == 0) {
            hasPreferenceSelected = false;
          }
        }
        if(Object.keys(roleDict).length === this.unified_frame_arguments.length && hasPreferenceSelected) {
          //all roles are set, and are uniq
          //TODO: aktywne preferencje w argumencie nie znajdują się w relacji hipo-/hiperonimii.
          const data = {'unified_frame_id': this.unified_frame.id};
          $.ajax({
            type: 'post',
            url: '/' + lang + '/unifier/change_unified_frame_status_to_ready/',
            dataType: 'json',
            data: data,
            timeout: 60000,
            success: function (response) {
              show_info('Status ramy został zmieniony');
              this.$emit('goToDisplay')
            }.bind(this),
            error: function (request, errorType, errorMessage) {
              show_error(errorType + ' (' + errorMessage + ')');
            }
          });
        } else {
          alert(gettext("Role dla wszystkich argumentów powinny być ustawione oraz unikalne. Preferencje selekcyjne dla wszystkich argumentów powinny być ustawione."));
        }
      }
    },
    change_slowal2unified_frame_argument_mapping(slowal_frame) {
      if(slowal_frame === this.active_slowal_frame && this.selectedFrameArguments && this.selectedFrameArguments.length == 2) {
        const data = {
          'unified_frame_id': this.unified_frame.id,
          'slowal_frame_id': slowal_frame.id,
          'slowal_frame_selected_arguments': JSON.stringify(this.selectedFrameArguments.map(arg => arg.argument_id))
        };
        $.ajax({
          type: 'post',
          url: '/' + lang + '/unifier/change_slowal2unified_fram_argument_mapping/',
          dataType: 'json',
          data: data,
          timeout: 60000,
          success: function (response) {
            show_info('Argumenty w ramie zostały zmienione');
            this.loadFrame();
          }.bind(this),
          error: function (request, errorType, errorMessage) {
            show_error(errorType + ' (' + errorMessage + ')');
          }
        });
      } else {
        alert(gettext("Zamiany pozycji argumentu w ramie wymaga zaznaczenia dokładnie 2 argumentów."));
      }
    },
    isSuperLeksykograf() {
      return has_permission("users.view_assignment");
    },
    isFrameVerified(frame) {
      const isSuperLeksykograf = this.isSuperLeksykograf();
      return (!isSuperLeksykograf && frame.status === 'G') || (isSuperLeksykograf && frame.status === 'S')
    },
    select_slowal_frame_req(to_invoke) {
      if(this.active_slowal_frame) {
        to_invoke();
      } else {
        alert(gettext("Wybierz ramę, dla której chcesz zmienić status."));
      }
    },
    change_slowal_frame_status(status) {
      this.select_slowal_frame_req(() => {
        let frame = this.active_slowal_frame;

        if(status === 'B') {
          this.setup_notes_slowal_frame_with_title_and_body("Niedopasowana jednostka (identyfikator jednostki)", "Do zatwierdzenia uznanie ramy Walentego jednostek (lista jednostek)" +
              "za niedopasowana do zunifikowanej ramy (identyfikator) przez (Leksykograf).");
        } else if(status === 'Z') {
          this.setup_notes_slowal_frame_with_title_and_body("Błędna jednostka (identyfikator jednostki)", "Do zatwierdzenia uznanie ramy Walentego jednostek (lista jednostek)" +
              "za niedopasowana do zunifikowanej ramy (identyfikator) przez (Leksykograf).");
        }

        const data = {
          'unified_frame_id': this.unified_frame.id,
          'slowal_frame_id': frame.id,
          'status': status
        };
        $.ajax({
          type: 'post',
          url: '/' + lang + '/unifier/change_slowal_frame_status/',
          dataType: 'json',
          data: data,
          timeout: 60000,
          success: function (response) {
            show_info('Status ramy został zmieniony');
            this.$emit('refreshEntriesList');
            this.loadFrame();
          }.bind(this),
          error: function (request, errorType, errorMessage) {
            show_error(errorType + ' (' + errorMessage + ')');
          }
        });
      });
    },
    slowal_frame_ready_rollback(status) {
      this.select_slowal_frame_req(() => {
        let frame = this.active_slowal_frame;
        const isSuperLeksykograf = this.isSuperLeksykograf();
        let status = null;
        if (!isSuperLeksykograf && frame.status === 'G') {
          //przywracamy O
          status = 'O';
        } else if (!isSuperLeksykograf && frame.status === 'O') {
          //ustawiany na Gotowe
          status = 'G';
        } else if (isSuperLeksykograf && frame.status === 'S') {
          //ustawiany Sprawdzone
          status = 'G';
        } else {
          status = 'S';
        }

        this.change_slowal_frame_status(status);
      });
    },
    isFrameVisible(status) {
      return (status != 'B' && status != 'C') || this.isSuperLeksykograf();
    },
    changeStatusButtonTitleToDefault() {
      const isSuperLeksykograf = this.isSuperLeksykograf();
      this.statusButtonTitle = isSuperLeksykograf ? 'Sprawdź' : 'Gotowe';
    },
    deselectSlowalFrameSelectedElements () {
      this.subentries.forEach(subentry => {
        subentry.schemata.forEach(s => {
          s.selected = false;
        });
      });
      this.frames.forEach(frame => {
        frame.lexical_units.forEach(lu => {
          lu.selected = false;
        });
        frame.arguments.forEach(argument => {
          argument.selected = false;
        });
      });
      this.selectedLus = [];
      this.selectedFrameArguments = [];
      this.selectedSchemas = [];
    },
    slowalFrameSelected(frame) {
      this.deselectSlowalFrameSelectedElements();
      if (this.active_slowal_frame === frame) {
        this.active_slowal_frame = null;
        if(!this.readOnly) {
          this.setup_notes_unified_frame();
          this.changeStatusButtonTitleToDefault();
        }
      } else {
        this.active_slowal_frame = frame;
        if(!this.readOnly) {
          this.setup_notes_slowal_frame()
          if (this.isFrameVerified(frame)) {
            this.statusButtonTitle = 'Przywróć';
          }
        }
      }
    },
    isSelectedFrame(frame) {
      if(this.active_slowal_frame) {
        return frame.id === this.active_slowal_frame.id;
      } else {
        return false;
      }
    },
    extract_frames_to_new_frame() {

      const existingSelect = function () {
        return this.frames.map(frame => {
          return `<label><input type="checkbox" name="frames" value="${frame.id}" /> ${lexical_units2dom(frame.lexical_units)}</label><br />`;
        }).join("");
      }.bind(this);

      const extract_frames_to_new_frame_popup = {
        state0: {
          title: 'Podaj nazwę zduplikowanej ramy',
          html: existingSelect,
          buttons: {Anuluj: 0, Wydziel: 1},
          focus: -1,
          submit: function (e, v, m, f) {
            if (v == 0) {
              e.preventDefault();
              $.prompt.close();
            }
            if (v === 1) {
              e.preventDefault();
              let frame_ids = normalizeFormData(f.frames);
              const data = { 'unified_frame_id' : this.unified_frame.id, 'slowal_frame_ids' : JSON.stringify(frame_ids),
                'target_unified_frame_id': ''};
              $.ajax({
                type     : 'post',
                url      : '/' + lang + '/unifier/extract_frames_to_new_frame/',
                dataType : 'json',
                data     : data,
                timeout  : 60000,
                success  : function(response) {
                  show_info('Ramy zostały wydzielone do nowej ramy zunifikowanej.');
                  this.loadFrame();
                  let newUnifiedFrameId = response.unified_frame_id;
                  this.currentPreviewedUnifiedFrameId = newUnifiedFrameId;
                  $.prompt.close();
                }.bind(this),
                error: function(request, errorType, errorMessage) {
                  show_error(errorType + ' (' + errorMessage + ')');
                  $.prompt.close();
                }
              });
            }
          }.bind(this)
        }
      }
      $.prompt(extract_frames_to_new_frame_popup);
    },
    extract_frame_to_preview_frame() {
      if(this.currentPreviewedUnifiedFrameId !== -1 && this.currentPreviewedUnifiedFrameId !== this.unified_frame.id) {
        let target_unified_frame_id = this.currentPreviewedUnifiedFrameId;
        const data = {
          'unified_frame_id': this.unified_frame.id, 'slowal_frame_ids': JSON.stringify([this.active_slowal_frame.id]),
          'target_unified_frame_id': target_unified_frame_id
        };
        $.ajax({
          type: 'post',
          url: '/' + lang + '/unifier/extract_frames_to_new_frame/',
          dataType: 'json',
          data: data,
          timeout: 60000,
          success: function (response) {
            show_info('Zaznaczona rama została przeniosiona.');
            this.active_slowal_frame = null;
            this.loadFrame();
            this.internalForceRefresh += 1;
          }.bind(this),
          error: function (request, errorType, errorMessage) {
            show_error(errorType + ' (' + errorMessage + ')');
          }
        });
      }
    },
    changePreviewedUnifiedFrameId (unifiedFrameId) {
      this.currentPreviewedUnifiedFrameId = unifiedFrameId;
    },
    swapUnifiedFrames () {
      if (this.currentPreviewedUnifiedFrameId !== -1 && this.unified_frame.id !== this.currentPreviewedUnifiedFrameId) {
        this.$emit("swapFrames", this.currentPreviewedUnifiedFrameId);
      }
    },
    changeShowVerifiedFrames (val) {
      this.showVerifiedFrames = val;
      this.hidden_frames = [];
    },
    getArgumentCSS(argument) {
      return (argument.role ? argument.role.str + ' ' : '') + (argument == this.active_unified_frame_argument ? 'active' : '');
    },
    insideFrameSelectionChanged (selectedFrameArguments, selectedLus) {
      this.selectedLus = selectedLus;
      this.selectedFrameArguments = selectedFrameArguments;
    },
    schemataSelected(schemas) {
      this.selectedSchemas = schemas;
    },
    deleteUnifiedFrames() {
      if (!confirm(gettext("Czy na pewno chcesz zunifikowaną ramę?"))) return false;
      $.ajax({
        type: 'post',
        url: '/' + lang + `/unifier/delete_unified_frame/${this.unified_frame.id}/`,
        dataType: 'json',
        timeout: 60000,
        success: function (response) {
          show_info('Zunifikowana rama została usunięta.');
          this.$emit('refreshEntriesList');
          this.$emit("swapFrames", null);
        }.bind(this),
        error: function (request, errorType, errorMessage) {
          show_error(errorType + ' (' + errorMessage + ')');
        }
      });
    },
    isReadOnlyForSuperLeksykograf () {
      return this.isSuperLeksykograf() && this.unified_frame.status === 'O';
    },
    hideFrame (frame) {
      this.hidden_frames.push(frame);
    }
  },
  mounted() {

    if(this.unifiedFrameId) {
      this.loadFrame();
    }

    this.changeStatusButtonTitleToDefault();
    if(!this.readOnly) {
      Split(['#semantics-frames-pane', '#semantics-schemata-pane'], {
        sizes: [40, 60],
        minSize: 10,
        gutterSize: 4,
        elementStyle: (dimension, size, gutterSize) => {
          return {
            'flex-basis': 'calc(' + size + '% - ' + gutterSize + 'px)'
          }
        },
      });
      Split(['#semantics-unified-frame-pane', '#semantics-slowal-frames-pane', '#examples'], {
        sizes: [40, 40, 20],
        direction: 'vertical',
        gutterSize: 4,
        minSize: 10,
      });
    } else {
      Split(['#semantics-unified-frame-pane-preview', '#semantics-slowal-frames-pane-preview', '#examples-preview'], {
        sizes: [40, 40, 20],
        direction: 'vertical',
        gutterSize: 4,
        minSize: 10,
      });
    }
  },
  template: `
    <div class="col h-100 px-0 pt-0 pb-0 overflow-auto" id="semantics-frames-pane">
      <div :id="'semantics-unified-frame-pane' + (readOnly ? '-preview' : '')" class="col w-100 p-0 overflow-auto">
        <table v-if="!readOnly && !isReadOnlyForSuperLeksykograf()" class="table-button-menu sticky-top" cellspacing="1">
          <tr style="background-color: white;">
            <td id="change-title" @click="changeTitle" style="padding: 10px 15px 10px 15px; color: #000000;">Zmień nazwę</td>
            <td id="add-arg" @click="addArgument" style="padding: 10px 15px 10px 15px; color: #000000;">Dodaj argum.</td>
            <td style="padding: 10px 15px 10px 15px; color: #000000;" @click="addSelectivePreference">Dodaj prefer.</td>
            <td style="padding: 10px 15px 10px 15px; color: #000000;" @click="changeUnifiedFrameStatusToReady">Gotowe</td>
            <td id="duplicates" @click="duplicate" style="padding: 10px 15px 10px 15px; color: #000000;">Duplikuj</td>
          </tr>
          <tr style="background-color: white;">
            <td id="change-role" @click="changeRole" style="padding: 10px 15px 10px 15px; color: #000000;">Zmień rolę</td>
            <td id="remove-arg" @click="removeArgument" style="padding: 10px 15px 10px 15px; color: #000000;">Usuń argum.</td>
            <td id="change-windows" style="padding: 10px 15px 10px 15px; color: #000000;" @click="removeSelectionalPreference">Usuń prefer.</td>
            <td id="change-windows" style="padding: 10px 15px 10px 15px; color: #000000;" @click="swapUnifiedFrames">Zamień okna</td>
            <td v-if="frames.length == 0" id="change-windows" style="padding: 10px 15px 10px 15px; color: #000000;" @click="deleteUnifiedFrames">Usuń</td>
          </tr>
        </table>
        <spinner />
        <div align="center">
        <div align="left" style="display: table;">
          <div class="unifiedFrame mt-3" v-bind:data-frame_id="unified_frame.id" id="lexical-unit" v-html="unified_frame_title"></div>
          <table v-if="unified_frame.id" id="unified-frame" class="m-0 table-borderless border border-secondary text-dark frame active">
          <tbody>
            <tr>
              <th scope="row" class="py-2 px-1 text-secondary role-header">Role</th>
              
              <template v-for="argument in unified_frame_arguments">
              <td
                class="argument py-2 px-1 border-top border-left border-secondary role-column"
                    :class="getArgumentCSS(argument)"
                    @click="unifiedFrameArgumentSelected(argument)"
                    @mouseover="unifiedFrameArgumentHovered(argument)"
                    @mouseleave="unifiedFrameArgumentHovered(null)"
                >
                {{ argument.role_type }}
                
                <div
                  v-if="argument.role"
                >
                  [{{ argument.role.str }}]
                </div>
                <div v-else>
                  <ul class="ul-role">
                    <li v-for="proposed_role in argument.proposed_roles">
                      {{ proposed_role.str }}
                    </li>
                  </ul>
                </div>
              </td>
              </template>
              
              
            </tr>
            <tr>
              <th scope="row" class="py-0 px-1 text-secondary role-header">Selectional preferences</th>
              <td class="preferences py-0 px-0 border-top border-left border-secondary role-column"
                  v-for='argument in unified_frame_arguments'
                  :key='argument.id'
              >
                <ul class="ul-preference" v-if="argument.preferences.length > 0">
                  <li v-for='preference in argument.preferences'>
                    <div
                      v-if="preference.url != null"
                      class="preference py-2 px-1 preference-bold"
                    >
                      <a class="synset-plwn" v-bind:href="preference.url" target="_blank">{{ preference.str }}</a>
                    </div>
                    <div v-else class="preference py-2 px-1 preference-bold">{{ preference.str }}</div>
                  </li>
                </ul>
                <ul class="ul-preference">
                  <li v-for="preference in slowal_frames2selecional_preferencies_mapping[argument.id]">
                    <span v-if="preference.url != null" class="preference py-2 px-1">
                      <a class="synset-plwn" v-bind:href="preference.url" target="_blank">{{ preference.str }}</a>
                    </span>
                    <span v-else class="preference py-2 px-1">{{ preference.str }}</span>
                    <info-tooltip v-if="preference.info" :text="preference.info" />
                  </li>
                </ul>
              </td>
            </tr>
          </tbody>
        </table>
        </div>
        <div v-if="unified_frame.id" class="lu-table mt-3 mb-3">
            <table class="m-0 table-borderless border border-secondary text-dark">
              <tbody>
                <tr>
                  <th scope="row" class="py-2 px-1 text-secondary">Jednostka leksykalna</th>
                  <th scope="row" class="py-2 px-1 text-secondary">Opinion</th>
                </tr>
                <tr class="preferences py-0 px-0 border-top border-left border-secondary"
                  v-for='lexical_unit in lexical_units'
                >
                  <template v-if="isFrameVisible(lexical_unit.frame_status)">
                    <td class="argument py-2 px-1 border-top border-left border-secondary">{{ lexical_unit.str }}</td>
                    <td class="argument py-2 px-1 border-top border-left border-secondary">
                      <img v-bind:src="img_prefix + 'entries/img/' +lexical_unit.opinion_key + '.svg'" width="12" height="12" v-bind:alt="lexical_unit.opinion">
                      {{ lexical_unit.opinion }}
                    </td>
                    <td class="argument py-2 px-1 border-top border-left border-secondary">[{{ lexical_unit.frame_status }}]</td>
                  </template>
                </tr>
              </tbody>
            </table>
          </div>
      </div>
      </div>
      <div :id="'semantics-slowal-frames-pane' + (readOnly ? '-preview' : '')" class="col w-100 p-0 overflow-auto">
        <table v-if="!readOnly && unified_frame.id && !isReadOnlyForSuperLeksykograf()" class="table-button-menu sticky-top" cellspacing="1">
          <tr style="background-color: white;">
            <td id="wrong-frame" style="padding: 10px 15px 10px 15px; color: #000000;" @click="change_slowal_frame_status('B')">Błędna</td>
            <td id="hide-slowal-frame" style="padding: 10px 15px 10px 15px; color: #000000;"  @click="changeShowVerifiedFrames(false)">Ukryj gotowe</td>
            <td id="ready-slowal-frame" style="padding: 10px 15px 10px 15px; color: #000000;" @click="slowal_frame_ready_rollback">{{ statusButtonTitle }}</td>
            <td id="filter-slowal-frames" style="padding: 10px 15px 10px 15px; color: #000000;">Filtruj</td>
          </tr>
          <tr style="background-color: white;">
            <td id="inccorect-slowal-frame" style="padding: 10px 15px 10px 15px; color: #000000;" @click="change_slowal_frame_status('C')">
                Niepasująca
            </td>
            <td id="show-slowal-frame" style="padding: 10px 15px 10px 15px; color: #000000;" @click="changeShowVerifiedFrames(true)">
                Pokaż wszystkie {{this.hidden_frames.length > 0 ? '(ukrytych: ' + this.hidden_frames.length + ')' : ''}}
            </td>
            <td id="move-slowal-frame" style="padding: 10px 15px 10px 15px; color: #000000;"
                @click="extract_frame_to_preview_frame">
                Przerzuć
            </td>
          </tr>
        </table>
        <div id="semantics-frames">
          <template v-for="frame in frames">
              <div v-if="!hidden_frames.includes(frame)">
                <table class="table m-0 table-borderless border border-secondary text-dark"
                   v-if="(!isFrameVerified(frame) || showVerifiedFrames) && isFrameVisible(frame.status)"
                  >
                    <tbody>
                    <tr>
                      <td>
                        <div class="row">
                          <div class="col close" style="
                              cursor: pointer;
                              display: flex;
                              justify-content: flex-end;"
                              @click="hideFrame(frame)"
                          >&times;</div>
                        </div>
                      <div class="row">
                        <div
                          class="col"
                          @mouseenter="frame.hover=true"
                          @mouseleave="frame.hover=false"
                          @click="slowalFrameSelected(frame)"
                          :class="frame === active_slowal_frame ? 'active-frame' : frame.hover ? 'bg-highlight' : ''" 
                        >
                        <slowal-frame-component 
                            :frame="frame" :key="frame"
                            @frame-selection-changed="insideFrameSelectionChanged"
                         />
                     </div>
                    </div>
                  </td>
                    <th scope="row">
                      <div v-if="!readOnly && !isReadOnlyForSuperLeksykograf()">
                        <div v-if="isFrameVerified(frame)">
                          <input
                            type="checkbox"
                            :checked="isFrameVerified(frame)"
                            class="custom-control custom-checkbox"
                            disabled="disabled"
                          >
                        </div>
                        <br><br>
                        <button v-if="!isFrameVerified(frame)"
                          class="btn btn-sm btn-dark" role="button" 
                          @click="change_slowal2unified_frame_argument_mapping(frame)"
                          >Z</button>
                      </div>
                    </th>
                </tr>
                </tbody>
              </table>
              </div>
          </template>
        </div>
      </div>
      <div :id="'examples' + (readOnly ? '-preview' : '')" class="col w-100 p-0 tab-pane overflow-auto">
         <examples-component v-if="examples"
              :examples="examples" 
              :frame="active_slowal_frame"
              :frame_arguments="selectedFrameArguments"
              :lus="selectedLus"
              :schemas="selectedSchemas"
              :key="examples"
          />
<!--            <table id="semantics-examples" class="table table-sm table-hover">-->
<!--                <thead>-->
<!--                    <tr>-->
<!--                        <th scope="col">{{ gettext("Przykład") }}<i id="examples-argument"></i><i id="examples-lu"></i><i id="examples-schema"></i></th>-->
<!--                        <th scope="col">{{ gettext("Źródło") }}</th>-->
<!--                        <th scope="col">{{ gettext("Opinia") }}</th>-->
<!--                    </tr>-->
<!--                </thead>-->
<!--                <tbody id="semantics-examples-list">-->
<!--                </tbody>-->
<!--            </table>-->
<!--            <p class="mx-1 my-1" id="semantics-no-examples">{{ gettext("Brak przykładów") }}</p>-->
        </div>
    </div>
    <div v-if="!readOnly" class="col h-100 px-1 pt-0 pb-0 overflow-auto" style="padding-left: 0px!important; padding-right: 0px!important;" id="semantics-schemata-pane">
      <ul class="nav nav-pills nav-justified p-1" id="entryTabs" role="tablist">
        <li
          v-for="tab in right_pane_tabs"
          class="btn btn-sm btn-outline-dark nav-link mx-1"
          :class="right_pane_tab === tab.id && 'active'"
          @click="right_pane_tab = tab.id"
        >
          {{ tab.label }}
        </li>
      </ul>
      <div v-if="active_slowal_frame" :class="right_pane_tab !== 'schemata' && 'd-none'">
          <semantics-schemata-component 
                :subentries="subentries"
                :key="subentries"
                :frame="active_slowal_frame"
                :alternations="alternations"
                :realisation_phrases="realisation_phrases"
                :realisation_descriptions="realisation_descriptions"
                @schemata-selected="schemataSelected"
              />
      </div>
      <div v-if="right_pane_tab === 'frame_preview'" class="overflow-hidden h-100" >
        <frame-preview 
            :key="currentPreviewedUnifiedFrameId" 
            :initialUnifiedFrameId="currentPreviewedUnifiedFrameId" 
            @change-frame="changePreviewedUnifiedFrameId"
            @change-preview-to-edit="swapUnifiedFrames"
            :forceRefresh="internalForceRefresh" />
      </div>
      <div :class="right_pane_tab !== 'notes' && 'd-none'" id="notes-component"></div>
    </div>
  `
});

export default LexicalUnitEdit;
