from django import forms
from django.contrib.auth.models import Group, User
from django.urls import reverse_lazy
from django.utils.text import format_lazy
from django.utils.translation import gettext_lazy as _

from crispy_forms.helper import FormHelper
from crispy_forms.layout import HTML, Layout, Fieldset, ButtonHolder, Submit

from users.models import Note


class UserForm(forms.ModelForm):
    group = forms.ModelChoiceField(queryset=Group.objects.all(), label=_('Grupa'))

    class Meta:
        model = User
        fields = [
            'first_name',
            'last_name',
            'username',
            'email',
            'group',
            'is_active',
        ]

    def __init__(self, *args, instance, **kwargs):
        super().__init__(*args, instance=instance, **kwargs)
        self.helper = FormHelper()
        self.helper.layout = Layout(
            Fieldset('', 'first_name', 'last_name', 'username', 'email', 'group', 'is_active'),
            ButtonHolder(
                Submit('submit', _('Zapisz'), css_class='btn btn-sm btn-success'),
                HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('users:user_list'), _('Wróć'))),
            ),
        )
        for field in ['first_name', 'last_name', 'email']:
            self.fields[field].required = True
        if instance.pk:
            self.initial['group'] = instance.groups.first()

    def save(self, commit=True):
        instance = super().save(commit=commit)
        instance.groups.set([self.cleaned_data['group']])
        return instance


class UserProfileForm(forms.ModelForm):
    class Meta:
        model = User
        fields = [
            'first_name',
            'last_name',
            'email',
        ]

    def __init__(self, *args, instance, **kwargs):
        super().__init__(*args, instance=instance, **kwargs)
        self.helper = FormHelper()
        self.helper.layout = Layout(
            Fieldset('', 'first_name', 'last_name', 'email'),
            ButtonHolder(
                Submit('submit', _('Zapisz'), css_class='btn btn-sm btn-success'),
                HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('users:password_change'), _('Zmień swoje hasło'))),
                HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('dash'), _('Wróć'))),
            ),
        )
        for field in ['first_name', 'last_name', 'email']:
            self.fields[field].required = True


login_form_helper = FormHelper()
login_form_helper.layout = Layout(
    Fieldset('', 'username', 'password'),
    ButtonHolder(
        Submit('submit', _('Zaloguj się'), css_class='btn btn-sm btn-success'),
        HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('users:password_reset'), _('Nie pamiętam hasła'))),
    ),
)

password_reset_form_helper = FormHelper()
password_reset_form_helper.layout = Layout(
    Fieldset('', 'email'),
    ButtonHolder(
        Submit('submit', _('Zresetuj hasło'), css_class='btn btn-sm btn-success'),
        HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('users:login'), _('Wróć'))),
    )
)

password_change_form_helper = FormHelper()
password_change_form_helper.layout = Layout(
    Fieldset('', 'old_password', 'new_password1', 'new_password2'),
    ButtonHolder(
        Submit('submit', _('Zmień hasło'), css_class='btn btn-sm btn-success'),
        HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('users:user_profile'), _("Wróć"))),
    )
)

password_reset_set_password_form_helper = FormHelper()
password_reset_set_password_form_helper.layout = Layout(
    Fieldset('', 'new_password1', 'new_password2'),
    ButtonHolder(
        Submit('submit', _('Ustaw hasło'), css_class='btn btn-sm btn-success'),
        HTML(format_lazy('<a class="btn btn-sm btn-light" href="{}">{}</a>', reverse_lazy('dash'), _("Wróć"))),
    )
)


class NoteForm(forms.ModelForm):
    class Meta:
        model = Note
        fields = ["title", "note", "super", "for_all"]
