from django.db import models

class Subentry(models.Model):
    entry = models.ForeignKey('connections.Entry', related_name='subentries', on_delete=models.PROTECT)
    aspect = models.ForeignKey('Aspect', related_name='subentries', null=True,
                                         on_delete=models.PROTECT)
    inherent_sie = models.ForeignKey('InherentSie', related_name='subentries', null=True,
                                                    on_delete=models.PROTECT)
    negativity = models.ForeignKey('Negativity', related_name='subentries', null=True,
                                                 on_delete=models.PROTECT)
    predicativity = models.ForeignKey('Predicativity', related_name='subentries', null=True,
                                                       on_delete=models.PROTECT)
    schemata = models.ManyToManyField('Schema', related_name='subentries')
    schemata_count = models.PositiveIntegerField(null=False, default=0)

    def __str__(self):
        if self.inherent_sie is None:
            return '%s (%s,%s,%s)' % (self.entry.name, self.negativity, self.predicativity, self.aspect)
        return '%s %s (%s,%s,%s)' % (self.entry.name, self.inherent_sie, self.negativity, self.predicativity,
                                     self.aspect)


class Aspect(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class InherentSie(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class Negativity(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class Predicativity(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class Schema(models.Model):
    opinion = models.ForeignKey('SchemaOpinion', related_name='schemata', on_delete=models.PROTECT)
    phraseologic = models.BooleanField()
    positions = models.ManyToManyField('Position', related_name='schemata')
    positions_count = models.PositiveIntegerField(null=False, default=0)
    
    def sorted_positions(self):  # TODO: zaimplementowac wlasciwe sortowanie
        return self.positions.all()

    def __str__(self):
        return '%s: %s' % (self.opinion, ' + '.join([str(pos) for pos in self.sorted_positions()]))


class SchemaOpinion(models.Model):
    key = models.CharField(max_length=16, unique=True)
    # name = TODO: wymaga lokalizacji
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.key


class Position(models.Model):
    function = models.ForeignKey('SyntacticFunction', related_name='positions', null=True,
                                                      on_delete=models.PROTECT)
    control = models.ForeignKey('Control', related_name='positions', null=True,
                                           on_delete=models.PROTECT)
    pred_control = models.ForeignKey('PredicativeControl', related_name='positions', null=True,
                                                           on_delete=models.PROTECT)
    phrase_types = models.ManyToManyField('PhraseType', related_name='positions')
    phrases_count = models.PositiveIntegerField(null=False, default=0)
    
    def sorted_phrase_types(self):  # TODO: zaimplementowac wlasciwe sortowanie
        return self.phrase_types.all()

    def __str__(self):
        categories = []
        if self.function is not None:
            categories.append(str(self.function))
        if self.control is not None:
            categories.append(str(self.control))
        if self.pred_control is not None:
            categories.append(str(self.pred_control))
        return '%s{%s}' % (','.join(categories), ';'.join([str(pt) for pt in self.sorted_phrase_types()]))


class SyntacticFunction(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class Control(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class PredicativeControl(models.Model):
    name = models.CharField(max_length=16, unique=True)
    priority = models.PositiveIntegerField()

    class Meta:
        ordering = ['priority']

    def __str__(self):
        return self.name


class NaturalLanguageDescription(models.Model):
    lang = models.CharField(max_length=2, choices=[('pl', 'pl'), ('en', 'en')])
    negativity = models.ForeignKey(Negativity, null=True, on_delete=models.PROTECT)
    function = models.ForeignKey(SyntacticFunction, null=True, on_delete=models.PROTECT)
    control = models.ForeignKey(Control, null=True, on_delete=models.PROTECT)
    pred_control = models.ForeignKey(PredicativeControl, null=True, on_delete=models.PROTECT)
    # phrase_type replaced with phrase_str to reduce number of changes in importer
    #phrase_type = models.ForeignKey(PhraseType, null=False, related_name='descriptions',
    #                                            on_delete=models.PROTECT)
    phrase_str = models.TextField()
    description = models.TextField()

    class Meta:
        unique_together = ['negativity', 'function', 'control', 'pred_control', 'phrase_str', 'description']

    def __str__(self):
        return self.description
