from django.apps import apps
from django.contrib.auth.decorators import login_required, permission_required
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from django.contrib.sites.shortcuts import get_current_site
from django.http import JsonResponse
from django.shortcuts import get_object_or_404, render, redirect
from django.utils.translation import gettext_lazy as _
from django.views.decorators.http import require_http_methods

from common.decorators import ajax
from users.forms import UserForm, UserProfileForm, NoteForm
from users.models import Note
from users.utils import send_new_user_email


@permission_required('users.view_user')
def user_list(request):
    return render(request, 'user_list.html', {'users': User.objects.order_by('username')})


@login_required
def user_profile(request):
    if request.method == 'POST':
        form = UserProfileForm(instance=request.user, data=request.POST)
        if form.is_valid():
            form.save()
            return redirect('dash')
    else:
        form = UserProfileForm(instance=request.user)
    return render(request, 'user_profile.html', {'form': form})


@permission_required('users.add_user')
def user_add(request):
    if request.method == 'POST':
        form = UserForm(instance=User(), data=request.POST)
        if form.is_valid():
            user = form.save()
            send_new_user_email(site=get_current_site(request), user=user, use_https=request.is_secure())
            return redirect('users:user_list')
    else:
        form = UserForm(instance=User())
    return render(request, 'user_form.html', {'form': form, 'title': _('Dodaj użytkownika')})


@permission_required('users.change_user')
def user_edit(request, pk):
    user = get_object_or_404(User, pk=pk)
    if request.method == 'POST':
        form = UserForm(instance=user, data=request.POST)
        if form.is_valid():
            form.save()
            return redirect('users:user_list')
    else:
        form = UserForm(instance=user)
    return render(request, 'user_form.html', {'form': form, 'title': _('Edytuj użytkownika')})


@login_required
def get_notes(request, model, pk):
    model = apps.get_model(*model.split('.'))
    subject = get_object_or_404(model, pk=pk)
    ct = ContentType.objects.get_for_model(model)
    notes = Note.objects.filter(subject_ct=ct, subject_id=subject.pk).for_user(request.user).order_by('created_at')
    return JsonResponse({
        "notes": [{
            "pk": note.pk,
            "owner_label": note.owner_label,
            "created_at": note.created_at,
            "title": note.title,
            "note": note.note,
        } for note in notes],
    })


@require_http_methods(["POST"])
@login_required
def add_note(request, model, pk):
    model = apps.get_model(*model.split('.'))
    subject = get_object_or_404(model, pk=pk)
    note = Note(author=request.user, subject=subject)
    form = NoteForm(instance=note, data=request.POST)
    if form.is_valid():
        form.save()
        return JsonResponse({})
    return JsonResponse(form.errors.get_json_data(), status=400)


@require_http_methods(["DELETE"])
@login_required
def delete_note(request, pk):
    note = get_object_or_404(Note.objects.for_user(request.user), pk=pk)
    note.delete()
    return JsonResponse({})
