#ifndef LIBWCCL_OPS_FORMATTERS_H
#define LIBWCCL_OPS_FORMATTERS_H

#include <libwccl/ops/function.h>

namespace Wccl {

/**
 * Class that formats unary functions as string
 */
struct UnaryFunctionFormatter
{
	/** 
	 * @returns String representation of an unary function.
	 * It is in form of	
	 * operator_name(arg_str)	
	 * although the open and close brackets can be changed
	 * (some operators use [])
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const std::string& arg_str,
		const char* open_bracket = "(",
		const char* close_bracket = ")");

	/** 
	 * @returns String representation of an unary function.
	 * It is in form of
	 * operator_name(argument_expression_string)
	 * although the open and close brackets can be changed
	 * (some operators use [])
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const FunctionBase& arg_expr,
		const char* open_bracket = "(",
		const char* close_bracket = ")")
	{
		return to_string(
			tagset,
			f,
			arg_expr.to_string(tagset),
			open_bracket, close_bracket);
	}
	/** 
	 * @returns Raw string representation of an unary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_operator_name(arg_str)
	 * although the open and close brackets can be changed
	 * (some operators use [])
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const std::string& arg_str,
		const char* open_bracket = "(",
		const char* close_bracket = ")");

	/** 
	 * @returns Raw string representation of an unary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_operator_name(raw_argument_expression_string)
	 * although the open and close brackets can be changed
	 * (some operators use [])
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const FunctionBase& arg_expr,
		const char* open_bracket = "(",
		const char* close_bracket = ")")
	{
		return to_raw_string(
			f,
			arg_expr.to_raw_string(),
			open_bracket,
			close_bracket);
	}
};

/**
 * Class that formats binary functions as string
 */
struct BinaryFunctionFormatter
{
	/** 
	 * @returns String representation of a binary function.
	 * It is in form of
	 * operator_name(arg1_str, arg2_str)
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const std::string& arg1_str,
		const std::string& arg2_str);

	/** 
	 * @returns String representation of a binary function.
	 * It is in form of
	 * operator_name(arg1_str, arg2_expr_string)
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const std::string& arg1_str,
		const FunctionBase& arg2_expr)
	{
		return to_string(
			tagset,
			f,
			arg1_str,
			arg2_expr.to_string(tagset));
	}
	/** 
	 * @returns String representation of a binary function.
	 * It is in form of
	 * operator_name(arg1_expr_string, arg2_str)
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const FunctionBase& arg1_expr,
		const std::string& arg2_str)
	{
		return to_string(
			tagset,
			f,
			arg1_expr.to_string(tagset),
			arg2_str);
	}

	/** 
	 * @returns String representation of a binary function.
	 * It is in form of
	 * operator_name(arg1_expr_string, arg2_expr_string)
	 */
	static std::string to_string(
		const Corpus2::Tagset& tagset,
		const FunctionBase& f,
		const FunctionBase& arg1_expr,
		const FunctionBase& arg2_expr)
	{
		return to_string(
			tagset,
			f,
			arg1_expr.to_string(tagset),
			arg2_expr.to_string(tagset));
	}

	/** 
	 * @returns Raw string representation of a binary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_op_name(arg1_str, arg2_str)
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const std::string& arg1_str,
		const std::string& arg2_str);

	/** 
	 * @returns Raw string representation of a binary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_op_name(arg1_str, raw_arg2_expr_string)
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const std::string& arg1_str,
		const FunctionBase& arg2_expr)
	{
		return to_raw_string(
			f,
			arg1_str,
			arg2_expr.to_raw_string());
	}

	/** 
	 * @returns Raw string representation of a binary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_op_name(raw_arg1_expr_string, arg1_str)
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const FunctionBase& arg1_expr,
		const std::string& arg2_str)
	{
		return to_raw_string(
			f,
			arg1_expr.to_raw_string(),
			arg2_str);
	}

	/** 
	 * @returns Raw string representation of a binary function.
	 * Does not require tagset, may contain internal info
	 * and/or be incomplete. It is in form of
	 * raw_op_name(raw_arg1_expr_string, raw_arg2_expr_string)
	 */
	static std::string to_raw_string(
		const FunctionBase& f,
		const FunctionBase& arg1_expr,
		const FunctionBase& arg2_expr)
	{
		return to_raw_string(
			f,
			arg1_expr.to_raw_string(),
			arg2_expr.to_raw_string());
	}
};

} /* end ns Wccl */


#endif // LIBWCCL_OPS_FORMATTERS_H
