#include <boost/test/unit_test.hpp>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <libcorpus2/sentence.h>

#include <libwccl/ops/functions/constant.h>
#include <libwccl/ops/functions/bool/predicates/isinside.h>
#include <libwccl/ops/functions/bool/predicates/isoutside.h>

using namespace Wccl;

BOOST_AUTO_TEST_SUITE(position_predicates)

struct PosPredFix
{
	PosPredFix()
		: s(boost::make_shared<Corpus2::Sentence>()),
		  sc(s),
		  tagset(),
		  cx(sc, boost::make_shared<Variables>()),
		  pos_one(1),
		  pos_minus_one(-1),
		  nowhere(Position::Nowhere),
		  begin(Position::Begin),
		  end(Position::End),
		  pos_one_constant(new Constant<Position>(pos_one)),
		  pos_minus_one_constant(new Constant<Position>(pos_minus_one)),
		  nowhere_constant(new Constant<Position>(nowhere)),
		  begin_constant(new Constant<Position>(begin)),
		  end_constant(new Constant<Position>(end))
	{
		Corpus2::Token* the_token = new Corpus2::Token("ZZ", PwrNlp::Whitespace::ManySpaces);
		Corpus2::Tag t1(Corpus2::mask_t(0));
		Corpus2::Lexeme l1("aaa", t1);
		Corpus2::Lexeme l2("bbb", t1);
		the_token->add_lexeme(l1);
		the_token->add_lexeme(l2);
		s->append(the_token);
		s->append(the_token->clone());
	}

	boost::shared_ptr<Corpus2::Sentence> s;
	SentenceContext sc;
	Corpus2::Tagset tagset;

	FunExecContext cx;
	Position pos_one;
	Position pos_minus_one;
	Position nowhere;
	Position begin;
	Position end;
	boost::shared_ptr<Function<Position> > pos_one_constant;
	boost::shared_ptr<Function<Position> > pos_minus_one_constant;
	boost::shared_ptr<Function<Position> > nowhere_constant;
	boost::shared_ptr<Function<Position> > begin_constant;
	boost::shared_ptr<Function<Position> > end_constant;

};

BOOST_FIXTURE_TEST_CASE(is_inside_1, PosPredFix)
{
	IsInside is_inside(pos_one_constant);
	BOOST_CHECK(is_inside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(!is_inside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_inside_minus1, PosPredFix)
{
	IsInside is_inside(pos_minus_one_constant);
	BOOST_CHECK(!is_inside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(is_inside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_inside_nowhere, PosPredFix)
{
	IsInside is_inside(nowhere_constant);
	BOOST_CHECK(!is_inside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(!is_inside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_inside_begin, PosPredFix)
{
	IsInside is_inside(begin_constant);
	BOOST_CHECK(is_inside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(is_inside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_inside_end, PosPredFix)
{
	IsInside is_inside(end_constant);
	BOOST_CHECK(is_inside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(is_inside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_outside_1, PosPredFix)
{
	IsOutside is_outside(pos_one_constant);
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(is_outside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_outside_minus1, PosPredFix)
{
	IsOutside is_outside(pos_minus_one_constant);
	BOOST_CHECK(is_outside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_outside_nowhere, PosPredFix)
{
	IsOutside is_outside(nowhere_constant);
	BOOST_CHECK(is_outside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(is_outside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_outside_begin, PosPredFix)
{
	IsOutside is_outside(begin_constant);
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
}

BOOST_FIXTURE_TEST_CASE(is_outside_end, PosPredFix)
{
	IsOutside is_outside(end_constant);
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
	sc.advance();
	BOOST_CHECK(!is_outside.apply(cx)->get_value());
}
//------ to_string test cases -------

BOOST_FIXTURE_TEST_CASE(is_inside_to_string, PosPredFix)
{
	IsInside is_inside(end_constant);
	BOOST_CHECK_EQUAL("inside(end)", is_inside.to_string(tagset));
}

BOOST_FIXTURE_TEST_CASE(is_inside_to_raw_string, PosPredFix)
{
	IsInside is_inside(end_constant);
	BOOST_CHECK_EQUAL("inside(end)", is_inside.to_raw_string());
}

BOOST_FIXTURE_TEST_CASE(is_outside_to_string, PosPredFix)
{
	IsOutside is_outside(begin_constant);
	BOOST_CHECK_EQUAL("outside(begin)", is_outside.to_string(tagset));
}

BOOST_FIXTURE_TEST_CASE(is_outside_to_raw_string, PosPredFix)
{
	IsOutside is_outside(nowhere_constant);
	BOOST_CHECK_EQUAL("outside(nowhere)", is_outside.to_raw_string());
}
BOOST_AUTO_TEST_SUITE_END()
