#ifndef LIBWCCL_OPS_REGEX_H
#define LIBWCCL_OPS_REGEX_H

#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>
#include <unicode/regex.h>

#include <libwccl/exception.h>
#include <libwccl/values/bool.h>
#include <libwccl/values/strset.h>
#include <libwccl/ops/predicate.h>

namespace Wccl {

/**
 * Operator that applies a regular expression to a set of
 * strings and returns true if all of the strings match the
 * expression
 */
class Regex : public Predicate {
public:
	typedef boost::shared_ptr<Function<StrSet> > StrSetFunctionPtr;

	Regex(const StrSetFunctionPtr& strset_expr, const UnicodeString& patstr);

	/**
	 * String representation of the operator in form of:
	 * "regex(strset_expr_string, regex_string)"
	 */
	virtual std::string to_string(const Corpus2::Tagset& tagset) const;

	/**
	 * String representation of conditional operator in form of:
	 * "regex(strset_expr_raw_string, regex_string)"
	 * This version does not require tagset, but may be inclomplete
	 * and/or contain internal info.
	 */
	virtual std::string to_raw_string() const;

	virtual const std::string raw_operator_name() const {
		return "regex";
	}

protected:	
	typedef FunctionBase::BaseRetValPtr BaseRetValPtr;

	/**
	 * Get a string set from the argument expression,
	 * apply regular expression to each string one by one,
	 * return false if a string not matching expression is found.
	 * Return true if all strings matched the regular espression.
	 */
	virtual BaseRetValPtr apply_internal(const SentenceContext& context) const;

private:
	const StrSetFunctionPtr strset_expr_;
	const UnicodeString patstr_;
	const boost::shared_ptr<const RegexPattern> pattern_;
};

class RegexParseError : WcclError
{
public:
	RegexParseError(
		const UnicodeString& pattern,
		const UErrorCode& status_code,
		const UParseError& parse_error);

	std::string info() const;

	const std::string status;
	const int pattern_line;
	const int offset;
	const std::string pre_context;
	const std::string error;
	const std::string expression;
};

} /* end ns Wccl */


#endif // LIBWCCL_OPS_REGEX_H
