#ifndef LIBWCCL_OPS_FUNCTIONS_H
#define LIBWCCL_OPS_FUNCTIONS_H

#include <boost/shared_ptr.hpp>
#include <boost/assert.hpp>
#include <boost/mpl/assert.hpp>
#include <boost/type_traits/is_base_of.hpp>

#include <libwccl/ops/operator.h>
#include <libwccl/values/value.h>
#include <libwccl/sentencecontext.h>

namespace Wccl {

/**
 * Abstract base class for WCCL operators that are functions returning a Value
 */
class FunctionBase : public Operator {
protected:
	typedef boost::shared_ptr<Value> BaseRetValPtr;
	/**
	 * Applies the function, given the sentence context.
	 */
	virtual BaseRetValPtr apply_internal(const SentenceContext& context) const = 0;
};

/**
 * Abstract base class template for functional WCCL operators that are functions
 * returning a value of given type
 */
template<class TRet>
class Function : public FunctionBase {
	BOOST_MPL_ASSERT( (boost::is_base_of<Value, TRet>) );
	BOOST_MPL_ASSERT_NOT( (boost::is_same<Value, TRet>) );
public:
	/**
	 * Type returned after application of function (shared pointer to
	 * a variable of the specified return type)
	 */
	typedef boost::shared_ptr<TRet> RetValPtr;

	/**
	 * Applies the function, given the sentence context, returning specific
	 * type of Value (as shared pointer). Uses apply_internal which has to
	 * be specified in derived classes.
	 */
	RetValPtr apply(const SentenceContext& context) const {
		RetValPtr v = boost::dynamic_pointer_cast<TRet>(apply_internal(context));
		BOOST_ASSERT(v);
		return v;
	}
};

} /* end ns Wccl */

#endif // LIBWCCL_OPS_FUNCTIONS_H
