#ifndef LIBWCCL_OPS_FUNCTIONS_BOOL_PREDICATES_ISOUTSIDE_H
#define LIBWCCL_OPS_FUNCTIONS_BOOL_PREDICATES_ISOUTSIDE_H

#include <libwccl/ops/functions/bool/predicate.h>
#include <libwccl/ops/formatters.h>
#include <libwccl/values/position.h>

namespace Wccl {

/**
 * Predicate that checks if a position is outside of the sentence boundaries
 */
class IsOutside : public Predicate {
public:
	typedef boost::shared_ptr<Function<Position> > PosFunctionPtr;

	IsOutside(const PosFunctionPtr& pos_expr)
		: pos_expr_(pos_expr)
	{
		BOOST_ASSERT(pos_expr_);
	}

	/**
	 * @returns String representation of the function in the form of:
	 * "outside(arg_expr_string)"
	 */
	std::string to_string(const Corpus2::Tagset& tagset) const {
		return UnaryFunctionFormatter::to_string(tagset, *this, *pos_expr_);
	}

	/**
	 * @returns String representation of the function in the form of:
	 * "outside(arg_expr_raw_string)"
	 */
	std::string to_raw_string() const {
		return UnaryFunctionFormatter::to_raw_string(*this, *pos_expr_);
	}

	/**
	 * @returns Name of the function: "outside"
	 */
	std::string raw_name() const {
		return "outside";
	}

protected:
	const PosFunctionPtr pos_expr_;

	/**
	 * Takes values of position from argument, and checks if it is outside of the
	 * sentence boundaries, in the given context (i.e. relative to current position)
	 * @returns True value if position is outside of the sentence boundaries, False otherwise.
	 */
	BaseRetValPtr apply_internal(const FunExecContext& context) const {
		const boost::shared_ptr<const Position>& pos = pos_expr_->apply(context);
		return Predicate::evaluate(pos->is_outside(context.sentence_context()), context);
	}
};

} /* end ns Wccl */

#endif // LIBWCCL_OPS_FUNCTIONS_BOOL_PREDICATES_ISOUTSIDE_H
