#include <boost/test/unit_test.hpp>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <libcorpus2/sentence.h>

#include <libwccl/ops/functions/constant.h>
#include <libwccl/ops/functions/position/relativeposition.h>

using namespace Wccl;

BOOST_AUTO_TEST_SUITE(relative_position)

struct PosPredFix
{
	PosPredFix()
		: s(boost::make_shared<Corpus2::Sentence>()),
		  sc(s),
		  tagset(),
		  cx(sc, boost::make_shared<Variables>()),
		  pos_one(1),
		  pos_minus_one(-1),
		  nowhere(Position::Nowhere),
		  begin(Position::Begin),
		  end(Position::End),
		  pos_one_constant(new Constant<Position>(pos_one)),
		  pos_minus_one_constant(new Constant<Position>(pos_minus_one)),
		  nowhere_constant(new Constant<Position>(nowhere)),
		  begin_constant(new Constant<Position>(begin)),
		  end_constant(new Constant<Position>(end))
	{
		Corpus2::Token* the_token = new Corpus2::Token("ZZ", PwrNlp::Whitespace::ManySpaces);
		Corpus2::Tag t1(Corpus2::mask_t(0));
		Corpus2::Lexeme l1("aaa", t1);
		Corpus2::Lexeme l2("bbb", t1);
		the_token->add_lexeme(l1);
		the_token->add_lexeme(l2);
		s->append(the_token);
		s->append(the_token->clone());
	}

	boost::shared_ptr<Corpus2::Sentence> s;
	SentenceContext sc;
	Corpus2::Tagset tagset;

	FunExecContext cx;
	Position pos_one;
	Position pos_minus_one;
	Position nowhere;
	Position begin;
	Position end;
	boost::shared_ptr<Function<Position> > pos_one_constant;
	boost::shared_ptr<Function<Position> > pos_minus_one_constant;
	boost::shared_ptr<Function<Position> > nowhere_constant;
	boost::shared_ptr<Function<Position> > begin_constant;
	boost::shared_ptr<Function<Position> > end_constant;

};

BOOST_FIXTURE_TEST_CASE(rel_nowhere, PosPredFix)
{
	for(int offset = -2; offset < 3; offset++)
	{
		RelativePosition relpos(nowhere_constant, offset);
		BOOST_CHECK_EQUAL(
			Position::Nowhere,
			relpos.apply(cx)->get_value());
		sc.advance();
		BOOST_CHECK_EQUAL(
			Position::Nowhere,
			relpos.apply(cx)->get_value());
		sc.advance();
		BOOST_CHECK_EQUAL(
			Position::Nowhere,
			relpos.apply(cx)->get_value());
		sc.goto_start();
	}
}

BOOST_FIXTURE_TEST_CASE(rel_begin, PosPredFix)
{
	for(int offset = -2; offset < 3; offset++)
	{
		RelativePosition relpos(begin_constant, offset);
		BOOST_CHECK_EQUAL(
			offset,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.advance();
		BOOST_CHECK_EQUAL(
			offset,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.advance();
		BOOST_CHECK_EQUAL(
			offset,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.goto_start();
	}
}

BOOST_FIXTURE_TEST_CASE(rel_end, PosPredFix)
{
	for(int i = -2; i < 3; i++)
	{
		RelativePosition relpos(end_constant, i);
		BOOST_CHECK_EQUAL(
			sc.size() - 1 + i,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.advance();
		BOOST_CHECK_EQUAL(
			sc.size() - 1 + i,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.advance();
		BOOST_CHECK_EQUAL(
			sc.size() - 1 + i,
			cx.sentence_context().get_abs_position(*relpos.apply(cx)));
		sc.goto_start();
	}
}

BOOST_FIXTURE_TEST_CASE(rel_normal_pos, PosPredFix)
{
	for(int offset = -2; offset < 3; offset++)
	{
		for(int pos = -2; pos < 3; pos++)
		{
			boost::shared_ptr<Constant<Position> > position(new Constant<Position>(Position(pos)));
			RelativePosition relpos(position, offset);
			BOOST_CHECK_EQUAL(
				offset + pos,
				relpos.apply(cx)->get_value());
			sc.advance();
			BOOST_CHECK_EQUAL(
				offset + pos,
				relpos.apply(cx)->get_value());
			sc.advance();
			BOOST_CHECK_EQUAL(
				offset + pos,
				relpos.apply(cx)->get_value());
			sc.goto_start();
		}
	}
}

//------ to_string test cases -------

BOOST_FIXTURE_TEST_CASE(relpos_to_string, PosPredFix)
{
	RelativePosition relpos(begin_constant, 4);
	BOOST_CHECK_EQUAL("begin + 4", relpos.to_string(tagset));
	RelativePosition relpos_min1(pos_minus_one_constant, -1);
	BOOST_CHECK_EQUAL("-1 - 1", relpos_min1.to_string(tagset));
	RelativePosition relpos_zero(end_constant, 0);
	BOOST_CHECK_EQUAL("end + 0", relpos_zero.to_string(tagset));
}

BOOST_FIXTURE_TEST_CASE(relpos_to_raw_string, PosPredFix)
{
	RelativePosition relpos(nowhere_constant, 42);
	BOOST_CHECK_EQUAL("nowhere + 42", relpos.to_raw_string());
	RelativePosition relpos_min1(pos_one_constant, -1);
	BOOST_CHECK_EQUAL("1 - 1", relpos_min1.to_raw_string());
	RelativePosition relpos_zero(end_constant, 0);
	BOOST_CHECK_EQUAL("end + 0", relpos_zero.to_raw_string());
}
BOOST_AUTO_TEST_SUITE_END()
