#include <boost/test/unit_test.hpp>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <libcorpus2/sentence.h>
#include <libcorpus2/tagsetmanager.h>

#include <libwccl/ops/functions/constant.h>
#include <libwccl/ops/functions/tset/getsymbols.h>
#include <libwccl/ops/functions/tset/getsymbolsinrange.h>

using namespace Wccl;

BOOST_AUTO_TEST_SUITE(get_symbols_in_range_op)

struct SymbolsInRangeFix
{
	SymbolsInRangeFix()
		: s(boost::make_shared<Corpus2::Sentence>()),
		  sc(s),
		  tagset(Corpus2::get_named_tagset("kipi")),
		  cx(sc, boost::make_shared<Variables>()),
		  pos_zero(0),
		  pos_one(1),
		  pos_minus_one(-1),
		  nowhere(Position::Nowhere),
		  pos_zero_constant(new Constant<Position>(pos_zero)),
		  pos_one_constant(new Constant<Position>(pos_one)),
		  pos_minus_one_constant(new Constant<Position>(pos_minus_one)),
		  nowhere_constant(new Constant<Position>(nowhere)),
		  pos_plus_2_constant(new Constant<Position>(Position(2))),
		  pos_minus_2_constant(new Constant<Position>(Position(-2)))
	{
		Corpus2::Token* the_token = new Corpus2::Token(
				"One",
				PwrNlp::Whitespace::ManySpaces);
		Corpus2::Lexeme l1("aaa", tagset.parse_simple_tag("subst:sg:nom:m1", false));
		Corpus2::Lexeme l2("aaa", tagset.parse_simple_tag("subst:sg:nom:m2", false));
		the_token->add_lexeme(l1);
		the_token->add_lexeme(l2);
		s->append(the_token);
		Corpus2::Token* another_token = new Corpus2::Token(
				"Two",
				PwrNlp::Whitespace::ManySpaces);
		Corpus2::Lexeme l3("aaa", tagset.parse_simple_tag("subst:pl:dat:f", false));
		Corpus2::Lexeme l4("aaa", tagset.parse_simple_tag("prep:nom:wok", false));
		Corpus2::Lexeme l5("aaa", tagset.parse_simple_tag("adja", false));
		another_token->add_lexeme(l3);
		another_token->add_lexeme(l4);
		another_token->add_lexeme(l5);
		s->append(another_token);
		gnd = tagset.parse_symbol("gnd");
		nmb = tagset.parse_symbol("nmb");
		vcl = tagset.parse_symbol("vcl");
	}

	boost::shared_ptr<Corpus2::Sentence> s;
	SentenceContext sc;
	const Corpus2::Tagset& tagset;

	FunExecContext cx;
	Position pos_zero;
	Position pos_one;
	Position pos_minus_one;
	Position nowhere;
	boost::shared_ptr<Function<Position> > pos_zero_constant;
	boost::shared_ptr<Function<Position> > pos_one_constant;
	boost::shared_ptr<Function<Position> > pos_minus_one_constant;
	boost::shared_ptr<Function<Position> > nowhere_constant;
	boost::shared_ptr<Function<Position> > pos_plus_2_constant;
	boost::shared_ptr<Function<Position> > pos_minus_2_constant;
	TSet empty;
	Corpus2::Tag gnd;
	Corpus2::Tag nmb;
	Corpus2::Tag vcl;
	Corpus2::Tag pos;
};

BOOST_FIXTURE_TEST_CASE(range_nowhere, SymbolsInRangeFix)
{
	for(int i = 0; i < 3; i++)
	{
		GetSymbolsInRange range(gnd, nowhere_constant, pos_zero_constant);
		BOOST_CHECK(range.apply(cx)->equals(empty));
		GetSymbolsInRange r2(gnd, pos_zero_constant, nowhere_constant);
		BOOST_CHECK(r2.apply(cx)->equals(empty));
		GetSymbolsInRange r3(gnd, nowhere_constant, nowhere_constant);
		BOOST_CHECK(r3.apply(cx)->equals(empty));
		sc.advance();
	}
}

BOOST_FIXTURE_TEST_CASE(range_outside, SymbolsInRangeFix)
{
	GetSymbolsInRange range(gnd, pos_minus_2_constant, pos_minus_one_constant);
	BOOST_CHECK(range.apply(cx)->equals(empty));
	sc.advance();
	GetSymbolsInRange r2(gnd, pos_one_constant, pos_plus_2_constant);
	BOOST_CHECK(r2.apply(cx)->equals(empty));
}

BOOST_FIXTURE_TEST_CASE(range_valid_including_trimmed, SymbolsInRangeFix)
{
	GetSymbolsInRange range(gnd, pos_minus_2_constant, pos_zero_constant);
	BOOST_CHECK_EQUAL("{m1,m2}", range.apply(cx)->to_string(tagset));
	GetSymbolsInRange r2(gnd, pos_minus_one_constant, pos_zero_constant);
	BOOST_CHECK_EQUAL("{m1,m2}", r2.apply(cx)->to_string(tagset));
	GetSymbolsInRange r3(gnd, pos_one_constant, pos_plus_2_constant);
	BOOST_CHECK_EQUAL("{f}", r3.apply(cx)->to_string(tagset));
	GetSymbolsInRange r4(gnd, pos_zero_constant, pos_plus_2_constant);
	BOOST_CHECK_EQUAL("{m1,m2,f}", r4.apply(cx)->to_string(tagset));
	sc.advance();
	BOOST_CHECK_EQUAL("{m1,m2,f}", range.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{m1,m2,f}", r2.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{}", r3.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{f}", r4.apply(cx)->to_string(tagset));
	sc.advance();
	BOOST_CHECK_EQUAL("{m1,m2,f}", range.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{f}", r2.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{}", r3.apply(cx)->to_string(tagset));
	BOOST_CHECK_EQUAL("{}", r4.apply(cx)->to_string(tagset));
}

BOOST_FIXTURE_TEST_CASE(range_same_token, SymbolsInRangeFix)
{
	GetSymbolsInRange range(gnd, pos_zero_constant, pos_zero_constant);
	BOOST_CHECK_EQUAL("{m1,m2}", range.apply(cx)->to_string(tagset));
	sc.advance();
	BOOST_CHECK_EQUAL("{f}", range.apply(cx)->to_string(tagset));
	sc.advance();
	BOOST_CHECK_EQUAL("{}", range.apply(cx)->to_string(tagset));
}


BOOST_FIXTURE_TEST_CASE(range_to_string, SymbolsInRangeFix)
{
	GetSymbolsInRange range(gnd, pos_zero_constant, pos_plus_2_constant);
	BOOST_CHECK_EQUAL("range(gnd, 0, 2)", range.to_string(tagset));
}

BOOST_FIXTURE_TEST_CASE(range_to_raw_string, SymbolsInRangeFix)
{
	GetSymbolsInRange range(gnd, pos_zero_constant, pos_plus_2_constant);
	std::string expected = "range(" + gnd.raw_dump() + ", 0, 2)";
	BOOST_CHECK_EQUAL(expected, range.to_raw_string());
}

BOOST_AUTO_TEST_SUITE_END()
