#ifndef LIBWCCL_OPS_VARSETTER_H
#define LIBWCCL_OPS_VARSETTER_H

#include <libwccl/ops/predicate.h>
#include <libwccl/ops/formatters.h>

namespace Wccl {

/**
 * Operator that sets value of a variable of given type T
 * and returns True.
 */
template<class T>
class VarSetter : public Function<Bool> {
public:
	typedef typename boost::shared_ptr<Function<T> > ArgFunctionPtr;

	VarSetter(const VariableAccessor<T>& var_acc, const ArgFunctionPtr& arg_expr)
		: var_acc_(var_acc),
		  arg_expr_(arg_expr)
	{
		BOOST_ASSERT(arg_expr_);
	}
	
	/**
	 * @returns Function name for variable setter: "setvar"
	 */
	std::string raw_name() const {
		return "setvar";
	}

	/**
	 * @returns String representation of the variable setter which is
	 * setvar(var_repr, arg_expr_str)
	 * @note This version does not require a tagset, but may be incomplete
	 * and/or contain internal info.
	 */
	std::string to_raw_string() const {
		return BinaryFunctionFormatter::to_raw_string(
			*this,
			T::var_repr(var_acc_.get_name()),
			*arg_expr_);
	}

	/**
	 * @returns String representation of the variable setter which is
	 * setvar(var_repr, arg_raw_expr_str)
	 */
	std::string to_string(const Corpus2::Tagset& tagset) const {
		return BinaryFunctionFormatter::to_string(
			tagset,
			*this, 
			T::var_repr(var_acc_.get_name()),
			*arg_expr_);
	}
protected:
	/**
	 * Evaluate argument expression and assign the result to underlying variable.
	 * @returns True.
	 */
	BaseRetValPtr apply_internal(const FunExecContext& context) const {
		context.variables()->get_fast(var_acc_)->set_value(
			arg_expr_->apply(context)->get_value());
		return Predicate::True(context);
	}

private:
	const VariableAccessor<T> var_acc_;
	const ArgFunctionPtr arg_expr_;
};


} /* end ns Wccl */

#endif // LIBWCCL_OPS_VARSETTER_H
