#ifndef LIBWCCL_OPS_REGEX_H
#define LIBWCCL_OPS_REGEX_H

#include <unicode/regex.h>

#include <libwccl/exception.h>
#include <libwccl/values/strset.h>
#include <libwccl/ops/predicate.h>

namespace Wccl {

/**
 * Operator that applies a regular expression to a set of
 * strings and returns true if all of the strings match the
 * expression
 */
class Regex : public Predicate {
public:
	typedef boost::shared_ptr<Function<StrSet> > StrSetFunctionPtr;

	Regex(const StrSetFunctionPtr& strset_expr, const UnicodeString& patstr);

	/**
	 * @returns String representation of the function in form of:
	 * "regex(strset_expr_string, regex_string)"
	 */
	std::string to_string(const Corpus2::Tagset& tagset) const;

	/**
	 * @returns String representation of the function in form of:
	 * "regex(strset_expr_raw_string, regex_string)"
	 * @note This version does not require a tagset, but may be inclomplete
	 * and/or contain internal info.
	 */
	std::string to_raw_string() const;

	/**
	 * @returns Name of the function: "regex"
	 */
	std::string raw_name() const {
		return "regex";
	}

protected:	
	/**
	 * Get a string set from the argument expression,
	 * apply regular expression to each string one by one,
	 * return false if a string not matching expression is found.
	 * Return true if all strings matched the regular espression.
	 */
	BaseRetValPtr apply_internal(const FunExecContext& context) const;

private:
	const StrSetFunctionPtr strset_expr_;
	const UnicodeString patstr_;
	const boost::shared_ptr<const RegexPattern> pattern_;
};

class RegexParseError : WcclError
{
public:
	RegexParseError(
		const UnicodeString& pattern,
		const UErrorCode& status_code,
		const UParseError& parse_error);

	~RegexParseError() throw();
	
	std::string info() const;

	const std::string status;
	const int pattern_line;
	const int offset;
	const std::string pre_context;
	const std::string error;
	const std::string expression;
};

} /* end ns Wccl */


#endif // LIBWCCL_OPS_REGEX_H
