#ifndef LIBWCCL_OPS_FUNCTIONS_H
#define LIBWCCL_OPS_FUNCTIONS_H

#include <boost/shared_ptr.hpp>
#include <boost/assert.hpp>
#include <boost/mpl/assert.hpp>
#include <boost/type_traits/is_base_of.hpp>

#include <libwccl/ops/operator.h>
#include <libwccl/values/value.h>
#include <libwccl/ops/funexeccontext.h>

namespace Wccl {

/**
 * Abstract base class for WCCL operators that are functions returning a Value
 */
class FunctionBase : public Operator {
protected:
	typedef boost::shared_ptr<const Value> BaseRetValPtr;
	/**
	 * Applies the function for the given execution context.
	 */
	virtual BaseRetValPtr apply_internal(const FunExecContext& context) const = 0;
};

/**
 * Abstract base class template for functional WCCL operators that are functions
 * returning a value of given type
 */
template<class T>
class Function : public FunctionBase {
	BOOST_MPL_ASSERT( (boost::is_base_of<Value, T>) );
	BOOST_MPL_ASSERT_NOT( (boost::is_same<Value, T>) );
public:
	/**
	 * Type returned after application of function (shared pointer to
	 * a variable of the specified return type)
	 */
	typedef boost::shared_ptr<const T> RetValPtr;

	/**
	 * Applies the function, given the sentence context, returning specific
	 * type of Value (as shared pointer). Uses apply_internal which has to
	 * be specified in derived classes.
	 */
	RetValPtr apply(const FunExecContext& context) const {
		RetValPtr v = boost::dynamic_pointer_cast<const T>(apply_internal(context));
		BOOST_ASSERT(v);
		return v;
	}
};

} /* end ns Wccl */

#endif // LIBWCCL_OPS_FUNCTIONS_H
