#ifndef LIBWCCL_OPS_EQUALS_H
#define LIBWCCL_OPS_EQUALS_H

#include <boost/mpl/list.hpp>
#include <boost/mpl/count.hpp>
#include <libwccl/ops/predicate.h>
#include <libwccl/ops/formatters.h>

namespace Wccl {

/**
 * Predicate that checks for equality of values
 */
template <class T>
class Equals : public Predicate {
public:
	typedef boost::shared_ptr<Function<T> > ArgFunctionPtr;

	Equals(const ArgFunctionPtr& arg1_expr, const ArgFunctionPtr& arg2_expr)
		: arg1_expr_(arg1_expr), arg2_expr_(arg2_expr)
	{
		BOOST_ASSERT(arg1_expr_);
		BOOST_ASSERT(arg2_expr_);
	}

	virtual std::string to_string(const Corpus2::Tagset& tagset) const {
		return BinaryFunctionFormatter::to_string(tagset, *this, *arg1_expr_, *arg2_expr_);
	}

	virtual std::string to_raw_string() const {
		return BinaryFunctionFormatter::to_raw_string(*this, *arg1_expr_, *arg2_expr_);
	}

	virtual const std::string raw_operator_name() const {
		return "equals";
	}

protected:
	const ArgFunctionPtr arg1_expr_;
	const ArgFunctionPtr arg2_expr_;

	/**
	 * Take values of arguments from expressions and return True if they are equal,
	 * False otherwise.
	 */
	virtual BaseRetValPtr apply_internal(const FunExecContext& context) const {
		boost::shared_ptr<T> arg1 = this->arg1_expr_->apply(context);
		boost::shared_ptr<T> arg2 = this->arg2_expr_->apply(context);
		if(arg1->equals(*arg2)) {
			return Predicate::True(context);
		}
		return Predicate::False(context);
	}
};

/**
 * Predicate that checks for equality of Positions, given sentence context
 */
template <>
class Equals<Position> : public Predicate {
public:
	typedef boost::shared_ptr<Function<Position> > ArgFunctionPtr;

	Equals(const ArgFunctionPtr& arg1_expr, const ArgFunctionPtr& arg2_expr)
		: arg1_expr_(arg1_expr), arg2_expr_(arg2_expr)
	{
		BOOST_ASSERT(arg1_expr_);
		BOOST_ASSERT(arg2_expr_);
	}

	virtual std::string to_string(const Corpus2::Tagset& tagset) const {
		return BinaryFunctionFormatter::to_string(tagset, *this, *arg1_expr_, *arg2_expr_);
	}

	virtual std::string to_raw_string() const {
		return BinaryFunctionFormatter::to_raw_string(*this, *arg1_expr_, *arg2_expr_);
	}

	virtual const std::string raw_operator_name() const {
		return "equals";
	}

protected:
	const ArgFunctionPtr arg1_expr_;
	const ArgFunctionPtr arg2_expr_;

	typedef FunctionBase::BaseRetValPtr BaseRetValPtr;

	/**
	 * Take values of arguments from expressions and return True if they are equal,
	 * False otherwise.
	 */
	virtual BaseRetValPtr apply_internal(const FunExecContext& context) const {
		boost::shared_ptr<Position> arg1 = this->arg1_expr_->apply(context);
		boost::shared_ptr<Position> arg2 = this->arg2_expr_->apply(context);
		if(arg1->equals(*arg2)) {
			return Predicate::True(context);
		} else {
			//in the given context both positions can still point nowhere
			//even if they have different underlying value
			int abs_pos1 = context.get_abs_position(*arg1);
			int abs_pos2 = context.get_abs_position(*arg2);
			if(!context.is_inside(abs_pos1) && !context.is_inside(abs_pos2)) {
				return Predicate::True(context);
			}
		}
		return Predicate::False(context);
	}
};

} /* end ns Wccl */

#endif // LIBWCCL_OPS_EQUALS_H
